(function($){
  'use strict';

  // Variables globales para el mapa
  window.map = null;
  window.marker = null;
  window.autocomplete = null;
  window.mapInitialized = false;

  // Inicializar Google Maps
  window.initMap = function() {
    // Prevenir múltiples inicializaciones
    if (window.mapInitialized) return;
    
    if (!document.getElementById('property_map')) {
      console.log('Elemento del mapa no encontrado, posponiendo inicialización');
      return;
    }

    window.mapInitialized = true;

    // Coordenadas por defecto (Ciudad de México)
    var defaultLocation = { lat: 19.4326, lng: -99.1332 };

    // Inicializar mapa
    map = new google.maps.Map(document.getElementById('property_map'), {
      zoom: 13,
      center: defaultLocation,
      mapTypeControl: false,
      streetViewControl: false,
      fullscreenControl: false
    });

    // Inicializar marcador
    marker = new google.maps.Marker({
      map: map,
      draggable: true,
      position: defaultLocation
    });

    // Inicializar autocompletado
    var searchInput = document.getElementById('location_search');
    if (searchInput) {
      autocomplete = new google.maps.places.Autocomplete(searchInput, {
        types: ['address'],
        componentRestrictions: { country: 'MX' } // Restringir a México
      });

      // Cuando se selecciona un lugar
      autocomplete.addListener('place_changed', function() {
        var place = autocomplete.getPlace();
        
        if (!place.geometry) {
          console.log("No hay detalles geométricos para: '" + place.name + "'");
          return;
        }

        updateLocationData(place);
      });
    }

    // Evento cuando se arrastra el marcador
    marker.addListener('dragend', function() {
      var position = marker.getPosition();
      reverseGeocode(position.lat(), position.lng());
    });

    // Evento de clic en el mapa
    map.addListener('click', function(e) {
      marker.setPosition(e.latLng);
      reverseGeocode(e.latLng.lat(), e.latLng.lng());
    });
  };

  // Actualizar datos de ubicación basado en el lugar seleccionado
  function updateLocationData(place) {
    if (!place.geometry) return;

    // Centrar mapa y mover marcador
    if (place.geometry.viewport) {
      map.fitBounds(place.geometry.viewport);
    } else {
      map.setCenter(place.geometry.location);
      map.setZoom(17);
    }
    
    marker.setPosition(place.geometry.location);

    // Obtener componentes de la dirección
    var addressComponents = place.address_components || [];
    var locationData = {
      estado: '',
      ciudad: '',
      colonia: '',
      codigoPostal: '',
      direccion: place.formatted_address || '',
      lat: place.geometry.location.lat(),
      lng: place.geometry.location.lng()
    };

    // Procesar componentes de dirección
    for (var i = 0; i < addressComponents.length; i++) {
      var component = addressComponents[i];
      var types = component.types;

      if (types.includes('administrative_area_level_1')) {
        locationData.estado = component.long_name;
      } else if (types.includes('locality') || types.includes('administrative_area_level_2')) {
        locationData.ciudad = component.long_name;
      } else if (types.includes('sublocality') || types.includes('neighborhood')) {
        locationData.colonia = component.long_name;
      } else if (types.includes('postal_code')) {
        locationData.codigoPostal = component.long_name;
      }
    }

    // Llenar campos del formulario
    fillLocationFields(locationData);
  }

  // Geocodificación inversa (de coordenadas a dirección)
  function reverseGeocode(lat, lng) {
    var geocoder = new google.maps.Geocoder();
    var latlng = { lat: lat, lng: lng };

    geocoder.geocode({ location: latlng }, function(results, status) {
      if (status === 'OK' && results[0]) {
        var place = results[0];
        updateLocationData(place);
        $('#location_search').val(place.formatted_address);
      }
    });
  }

  // Llenar campos del formulario
  function fillLocationFields(data) {
    $('#estados').val(data.estado);
    $('#ciudad').val(data.ciudad);
    $('#colonia').val(data.colonia);
    $('#codigo_postal').val(data.codigoPostal);
    $('#location_specific').val(data.direccion);
    $('#latitud').val(data.lat.toFixed(6));
    $('#longitud').val(data.lng.toFixed(6));
  }

  function go(step, $form){
    var $steps = $form.find('.hpcp__step');
    var $ind = $form.find('.hpcp__step-indicator');
    $steps.removeClass('is-active');
    $ind.removeClass('is-active');
    var idx = step - 1;
    $steps.eq(idx).addClass('is-active');
    $ind.eq(idx).addClass('is-active');
    $form.data('hpcp-step', step);
    window.scrollTo({ top: $form.offset().top - 20, behavior: 'smooth' });
  }

  function validateStep(step, $form){
    // minimal client-side validation
    if(step === 1){
      var title = $form.find('input[name="post_title"]').val().trim();
      var content = $form.find('textarea[name="post_content"]').val().trim();
      var price = $form.find('input[name="precio"]').val().trim();
      var transaccion = $form.find('select[name="transaccion"]').val();
      var propertyType = $form.find('select[name="property-type"]').val();
      var estado = $form.find('input[name="estados"]').val().trim();
      var ciudad = $form.find('input[name="ciudad"]').val().trim();
      
      if(!title || !content || !price || !transaccion || !propertyType || !estado || !ciudad){
        alert('Completa todos los campos obligatorios: Título, Descripción, Precio, Tipo de transacción, Tipo de propiedad, Estado y Ciudad.');
        return false;
      }
    }
    if(step === 3){ // Actualizado de step 4 a step 3
      // Validar imagen destacada
      var featuredInput = $form.find('input[name="featured_image"]')[0];
      if(!featuredInput || !featuredInput.files || featuredInput.files.length === 0){
        alert('Debes seleccionar una imagen destacada.');
        return false;
      }
      
      // Validar tamaño de imagen destacada
      var maxSize = 2 * 1024 * 1024; // 2MB
      if(featuredInput.files[0].size > maxSize){
        alert('La imagen destacada debe ser menor a 2MB.');
        return false;
      }
      
      var yt = $form.find('input[name="visita_virtual"]').val().trim();
      if(yt){
        var re = /^(https?:\/\/)?(www\.)?(youtube\.com|youtu\.be)\/(watch\?v=|embed\/|shorts\/)?[\w\-]{6,}/i;
        if(!re.test(yt)){
          alert('La URL de visita virtual debe ser de YouTube.');
          return false;
        }
      }
      
      // Validar galería
      var galeriaInput = $form.find('input[name="galeria_propiedad[]"]')[0];
      if(galeriaInput && galeriaInput.files.length > 5){
        alert('Máximo 5 imágenes permitidas en la galería.');
        return false;
      }
      
      // Validar tamaño de imágenes de galería
      if(galeriaInput && galeriaInput.files){
        for(var i = 0; i < galeriaInput.files.length; i++){
          if(galeriaInput.files[i].size > maxSize){
            alert('Cada imagen debe ser menor a 2MB. La imagen "' + galeriaInput.files[i].name + '" es muy grande.');
            return false;
          }
        }
      }
    }
    return true;
  }

  $(document).on('click', '.hpcp__btn--next', function(){
    var $form = $(this).closest('form.hpcp');
    var step = Number($form.data('hpcp-step') || 1);
    if(!validateStep(step, $form)) return;
    go(step+1, $form);
  });

  $(document).on('click', '.hpcp__btn--prev', function(){
    var $form = $(this).closest('form.hpcp');
    var step = Number($form.data('hpcp-step') || 1);
    go(Math.max(1, step-1), $form);
  });

  $(function(){
    $('.hpcp').each(function(){ $(this).data('hpcp-step', 1); });
    
    // Verificar si Google Maps está disponible y el elemento del mapa existe
    if (typeof google !== 'undefined' && typeof google.maps !== 'undefined' && document.getElementById('property_map')) {
      // Si Google Maps ya está cargado pero no se ha inicializado
      if (!window.mapInitialized) {
        setTimeout(function() {
          if (typeof window.initMap === 'function') {
            window.initMap();
          }
        }, 100);
      }
    }
    
    // Manejar previsualización de imagen destacada
    $(document).on('change', '#featured_input', function(){
      var file = this.files[0];
      var preview = $('#featured_preview');
      preview.empty();
      
      if(file){
        var maxSize = 2 * 1024 * 1024; // 2MB
        
        if(file.size > maxSize){
          var errorDiv = $('<div class="hpcp__error-img">❌ Imagen muy grande - máximo 2MB</div>');
          preview.append(errorDiv);
          return;
        }
        
        if(file.type.startsWith('image/')){
          var reader = new FileReader();
          reader.onload = function(e){
            var imgDiv = $('<div class="hpcp__featured-img"></div>');
            var img = $('<img>').attr('src', e.target.result).css({
              'max-width': '200px',
              'max-height': '150px',
              'object-fit': 'cover',
              'border-radius': '8px',
              'border': '2px solid #16182D',
              'display': 'block',
              'margin': '8px 0'
            });
            var fileName = $('<small>').text(file.name).css({
              'color': '#6b7280',
              'display': 'block'
            });
            imgDiv.append(img, fileName);
            preview.append(imgDiv);
          };
          reader.readAsDataURL(file);
        }
      }
    });
    
    // Manejar previsualización de galería
    $(document).on('change', '#galeria_input', function(){
      var files = this.files;
      var preview = $('#galeria_preview');
      preview.empty();
      
      if(files.length > 5){
        alert('Máximo 5 imágenes permitidas. Se mostrarán solo las primeras 5.');
      }
      
      var maxFiles = Math.min(files.length, 5);
      var maxSize = 2 * 1024 * 1024; // 2MB
      
      for(var i = 0; i < maxFiles; i++){
        var file = files[i];
        
        if(file.size > maxSize){
          var errorDiv = $('<div class="hpcp__error-img">📷❌ ' + file.name + ' (muy grande - max 2MB)</div>');
          preview.append(errorDiv);
          continue;
        }
        
        if(file.type.startsWith('image/')){
          var reader = new FileReader();
          reader.onload = function(e){
            var imgDiv = $('<div class="hpcp__preview-img"></div>');
            var img = $('<img>').attr('src', e.target.result).css({
              'width': '100px',
              'height': '100px',
              'object-fit': 'cover',
              'border-radius': '8px',
              'border': '2px solid #e5e7eb'
            });
            imgDiv.append(img);
            preview.append(imgDiv);
          };
          reader.readAsDataURL(file);
        }
      }
    });
  });

})(jQuery);