<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class HPCP_Admin {

    public function __construct() {
        add_action( 'admin_menu', [ $this, 'add_admin_menu' ] );
        add_action( 'admin_init', [ $this, 'admin_init' ] );
    }

    public function add_admin_menu() {
        add_options_page(
            __('Hunter Price - Crear Propiedad', 'hp-crear-propiedad'),
            __('Crear Propiedad', 'hp-crear-propiedad'),
            'manage_options',
            'hpcp-settings',
            [ $this, 'admin_page' ]
        );
    }

    public function admin_init() {
        register_setting( 'hpcp_settings', 'hpcp_options' );

        add_settings_section(
            'hpcp_google_maps_section',
            __('Configuración de Google Maps', 'hp-crear-propiedad'),
            [ $this, 'google_maps_section_callback' ],
            'hpcp_settings'
        );

        add_settings_field(
            'google_maps_api_key',
            __('Google Maps API Key', 'hp-crear-propiedad'),
            [ $this, 'google_maps_api_key_callback' ],
            'hpcp_settings',
            'hpcp_google_maps_section'
        );

        add_settings_section(
            'hpcp_amenities_section',
            __('Configuración de Amenidades', 'hp-crear-propiedad'),
            [ $this, 'amenities_section_callback' ],
            'hpcp_settings'
        );

        add_settings_field(
            'available_amenities',
            __('Amenidades Disponibles', 'hp-crear-propiedad'),
            [ $this, 'available_amenities_callback' ],
            'hpcp_settings',
            'hpcp_amenities_section'
        );
    }

    public function google_maps_section_callback() {
        echo '<p>' . __('Configura tu clave de API de Google Maps para habilitar el mapa interactivo y autocompletado de direcciones.', 'hp-crear-propiedad') . '</p>';
        echo '<p><strong>' . __('APIs necesarias:', 'hp-crear-propiedad') . '</strong> Maps JavaScript API, Places API, Geocoding API</p>';
        echo '<p><a href="https://console.cloud.google.com/" target="_blank">' . __('Obtener API Key en Google Cloud Console', 'hp-crear-propiedad') . ' →</a></p>';
    }

    public function google_maps_api_key_callback() {
        $options = get_option( 'hpcp_options', [] );
        $api_key = isset( $options['google_maps_api_key'] ) ? $options['google_maps_api_key'] : '';
        
        echo '<input type="text" name="hpcp_options[google_maps_api_key]" value="' . esc_attr( $api_key ) . '" class="regular-text" placeholder="AIzaSyC..." />';
        echo '<p class="description">' . __('Ingresa tu clave de API de Google Maps. Déjalo vacío para deshabilitar el mapa.', 'hp-crear-propiedad') . '</p>';
        
        if ( ! empty( $api_key ) ) {
            echo '<p class="description" style="color: #46b450;">✓ ' . __('API Key configurada correctamente', 'hp-crear-propiedad') . '</p>';
        } else {
            echo '<p class="description" style="color: #dc3232;">⚠ ' . __('Sin API Key: el mapa no funcionará', 'hp-crear-propiedad') . '</p>';
        }
    }

    public function amenities_section_callback() {
        echo '<p>' . __('Configura las amenidades disponibles en el formulario. Una por línea.', 'hp-crear-propiedad') . '</p>';
    }

    public function available_amenities_callback() {
        $options = get_option( 'hpcp_options', [] );
        $amenities = isset( $options['available_amenities'] ) ? $options['available_amenities'] : '';
        
        // Valores por defecto si está vacío
        if ( empty( $amenities ) ) {
            $default_amenities = [
                'Wi-Fi',
                'Estacionamiento',
                'Gimnasio',
                'Canchas de basquetball',
                'Canchas de soccer',
                'Multicanchas',
                'Petfriendly',
                'Amueblado',
                'Servicios incluidos',
                'Caseta de vigilancia',
                'Piscina',
                'Área de juegos',
                'Jardín',
                'Terraza',
                'Balcón',
                'Aire acondicionado',
                'Calefacción',
                'Seguridad 24hrs'
            ];
            $amenities = implode( "\n", $default_amenities );
        }
        
        echo '<textarea name="hpcp_options[available_amenities]" rows="15" cols="50" class="large-text">' . esc_textarea( $amenities ) . '</textarea>';
        echo '<p class="description">' . __('Escribe una amenidad por línea. Estas aparecerán como opciones en el formulario.', 'hp-crear-propiedad') . '</p>';
    }

    public function admin_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Hunter Price - Crear Propiedad', 'hp-crear-propiedad'); ?></h1>
            
            <div class="notice notice-info">
                <h3><?php esc_html_e('Shortcodes Disponibles:', 'hp-crear-propiedad'); ?></h3>
                <p><strong><?php esc_html_e('Formulario de creación:', 'hp-crear-propiedad'); ?></strong> <code>[hp_crear_propiedad]</code></p>
                <p><?php esc_html_e('Usa este shortcode en cualquier página para mostrar el formulario de creación de propiedades.', 'hp-crear-propiedad'); ?></p>
                
                <p><strong><?php esc_html_e('Mapa de ubicación:', 'hp-crear-propiedad'); ?></strong> <code>[hp_desplegado_mapa]</code></p>
                <p><?php esc_html_e('Muestra un mapa con la ubicación de la propiedad actual. Úsalo en el single de propiedades.', 'hp-crear-propiedad'); ?></p>
                
                <h4><?php esc_html_e('Parámetros del mapa:', 'hp-crear-propiedad'); ?></h4>
                <ul>
                    <li><code>height="400px"</code> - <?php esc_html_e('Altura del mapa', 'hp-crear-propiedad'); ?></li>
                    <li><code>width="100%"</code> - <?php esc_html_e('Ancho del mapa', 'hp-crear-propiedad'); ?></li>
                    <li><code>zoom="15"</code> - <?php esc_html_e('Nivel de zoom (1-20)', 'hp-crear-propiedad'); ?></li>
                    <li><code>show_info="true"</code> - <?php esc_html_e('Mostrar información de ubicación', 'hp-crear-propiedad'); ?></li>
                </ul>
                
                <p><strong><?php esc_html_e('Ejemplo:', 'hp-crear-propiedad'); ?></strong> <code>[hp_desplegado_mapa height="300px" zoom="16" show_info="false"]</code></p>
            </div>

            <form method="post" action="options.php">
                <?php
                settings_fields( 'hpcp_settings' );
                do_settings_sections( 'hpcp_settings' );
                submit_button();
                ?>
            </form>

            <div class="hpcp-admin-info">
                <h2><?php esc_html_e('Información del Sistema', 'hp-crear-propiedad'); ?></h2>
                
                <table class="widefat" style="max-width: 600px;">
                    <tbody>
                        <tr>
                            <td><strong><?php esc_html_e('Versión del Plugin', 'hp-crear-propiedad'); ?></strong></td>
                            <td><?php echo defined('HPCP_VERSION') ? HPCP_VERSION : '1.0.0'; ?></td>
                        </tr>
                        <tr>
                            <td><strong><?php esc_html_e('Post Type', 'hp-crear-propiedad'); ?></strong></td>
                            <td>properties</td>
                        </tr>
                        <tr>
                            <td><strong><?php esc_html_e('Taxonomías requeridas', 'hp-crear-propiedad'); ?></strong></td>
                            <td>transaccion, property-type</td>
                        </tr>
                        <tr>
                            <td><strong><?php esc_html_e('Capacidades requeridas', 'hp-crear-propiedad'); ?></strong></td>
                            <td>edit_posts (usuario logueado)</td>
                        </tr>
                    </tbody>
                </table>

                <h3><?php esc_html_e('Estado del Mapa', 'hp-crear-propiedad'); ?></h3>
                <?php
                $options = get_option( 'hpcp_options', [] );
                $api_key = isset( $options['google_maps_api_key'] ) ? $options['google_maps_api_key'] : '';
                
                if ( ! empty( $api_key ) ) {
                    echo '<p style="color: #46b450;">✓ ' . __('Google Maps habilitado', 'hp-crear-propiedad') . '</p>';
                    echo '<p>' . __('El formulario incluirá mapa interactivo y autocompletado de direcciones.', 'hp-crear-propiedad') . '</p>';
                    echo '<p><small>' . __('📌 Los scripts se cargan solo cuando se usa el shortcode [hp_crear_propiedad] para optimizar el rendimiento.', 'hp-crear-propiedad') . '</small></p>';
                } else {
                    echo '<p style="color: #dc3232;">⚠ ' . __('Google Maps deshabilitado', 'hp-crear-propiedad') . '</p>';
                    echo '<p>' . __('Los campos de ubicación serán manuales. Configura la API Key arriba para habilitar el mapa.', 'hp-crear-propiedad') . '</p>';
                }
                ?>
            </div>

            <style>
                .hpcp-admin-info {
                    margin-top: 30px;
                    padding-top: 20px;
                    border-top: 1px solid #ddd;
                }
                .hpcp-admin-info table {
                    margin-top: 10px;
                }
                .hpcp-admin-info td {
                    padding: 8px 12px;
                }
            </style>
        </div>
        <?php
    }

    /**
     * Obtener la API Key de Google Maps
     */
    public static function get_google_maps_api_key() {
        $options = get_option( 'hpcp_options', [] );
        return isset( $options['google_maps_api_key'] ) ? trim( $options['google_maps_api_key'] ) : '';
    }

    /**
     * Obtener amenidades disponibles
     */
    public static function get_available_amenities() {
        $options = get_option( 'hpcp_options', [] );
        $amenities_text = isset( $options['available_amenities'] ) ? $options['available_amenities'] : '';
        
        if ( empty( $amenities_text ) ) {
            // Valores por defecto
            return [
                'Wi-Fi',
                'Estacionamiento',
                'Gimnasio',
                'Canchas de basquetball',
                'Canchas de soccer',
                'Multicanchas',
                'Petfriendly',
                'Amueblado',
                'Servicios incluidos',
                'Caseta de vigilancia',
                'Piscina',
                'Área de juegos',
                'Jardín',
                'Terraza',
                'Balcón',
                'Aire acondicionado',
                'Calefacción',
                'Seguridad 24hrs'
            ];
        }
        
        $amenities = array_map( 'trim', explode( "\n", $amenities_text ) );
        return array_filter( $amenities ); // Remover líneas vacías
    }
}

// Inicializar la clase admin
if ( is_admin() ) {
    new HPCP_Admin();
}