<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class HPCP_Form {

    public function __construct() {
        add_shortcode( 'hp_crear_propiedad', [ $this, 'render_form_shortcode' ] );
        add_shortcode( 'hp_desplegado_mapa', [ $this, 'render_map_shortcode' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_assets' ] );
        add_action( 'init', [ $this, 'handle_submission' ] );
    }

    public function enqueue_assets() {
        // Solo registrar los assets, no cargarlos automáticamente
        wp_register_style( 'hpcp-style', HPCP_URL . 'assets/css/hpcp.css', [], HPCP_VERSION );
        wp_register_script( 'hpcp-script', HPCP_URL . 'assets/js/hpcp.js', ['jquery'], HPCP_VERSION, true );
        wp_register_script( 'hpcp-display-map', HPCP_URL . 'assets/js/hpcp-display-map.js', ['jquery'], HPCP_VERSION, true );
        
        // Google Maps API - Solo registrar si hay API key configurada
        if ( class_exists( 'HPCP_Admin' ) ) {
            $api_key = HPCP_Admin::get_google_maps_api_key();
            if ( ! empty( $api_key ) ) {
                wp_register_script( 'google-maps', 
                    'https://maps.googleapis.com/maps/api/js?key=' . esc_attr( $api_key ) . '&libraries=places&callback=initMap&loading=async', 
                    [], null, true );
            }
        }
    }

    public function render_form_shortcode( $atts ) {
        if ( ! is_user_logged_in() ) {
            return '<p class="hpcp hpcp--notice">' . esc_html__( 'Debes iniciar sesión para crear propiedades.', 'hp-crear-propiedad' ) . '</p>';
        }
        if ( ! current_user_can( 'edit_posts' ) ) {
            return '<p class="hpcp hpcp--notice">' . esc_html__( 'No tienes permisos para crear propiedades.', 'hp-crear-propiedad' ) . '</p>';
        }

        // Cargar assets solo cuando se usa el shortcode
        wp_enqueue_style( 'hpcp-style' );
        wp_enqueue_script( 'hpcp-script' );
        
        // Solo cargar Google Maps si hay API key y el script está registrado
        if ( wp_script_is( 'google-maps', 'registered' ) ) {
            wp_enqueue_script( 'google-maps' );
        }

        $nonce = wp_create_nonce( 'hpcp_create_property' );

        ob_start(); ?>
        <form class="hpcp" id="hpcp-form" method="post" enctype="multipart/form-data">
            <input type="hidden" name="hpcp_action" value="create_property" />
            <input type="hidden" name="hpcp_nonce" value="<?php echo esc_attr( $nonce ); ?>" />

            <!-- Steps nav -->
            <ol class="hpcp__steps">
                <li class="hpcp__step-indicator is-active"><?php esc_html_e('Información básica','hp-crear-propiedad'); ?></li>
                <li class="hpcp__step-indicator"><?php esc_html_e('Detalles de la propiedad','hp-crear-propiedad'); ?></li>
                <li class="hpcp__step-indicator"><?php esc_html_e('Galería','hp-crear-propiedad'); ?></li>
                <li class="hpcp__step-indicator"><?php esc_html_e('Confirmar y guardar','hp-crear-propiedad'); ?></li>
            </ol>

            <!-- Step 1: Información básica -->
            <section class="hpcp__step is-active" data-step="1">
                <div class="hpcp__field">
                    <label class="hpcp__label"><?php esc_html_e('Título de la propiedad','hp-crear-propiedad'); ?> *</label>
                    <input class="hpcp__input" type="text" name="post_title" required />
                </div>

                <div class="hpcp__field">
                    <label class="hpcp__label"><?php esc_html_e('Descripción de la propiedad','hp-crear-propiedad'); ?> *</label>
                    <textarea class="hpcp__textarea hpcp__textarea--large" name="post_content" rows="8" required placeholder="<?php esc_attr_e('Describe las características principales de la propiedad...','hp-crear-propiedad'); ?>"></textarea>
                </div>

                <div class="hpcp__grid">
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Hot Deals','hp-crear-propiedad'); ?></label>
                        <select class="hpcp__select" name="_hot-deals">
                            <option value="No"><?php esc_html_e('No','hp-crear-propiedad'); ?></option>
                            <option value="Yes"><?php esc_html_e('Yes','hp-crear-propiedad'); ?></option>
                        </select>
                    </div>

                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Tipo de transacción','hp-crear-propiedad'); ?> *</label>
                        <select class="hpcp__select" name="transaccion" required>
                            <option value=""><?php esc_html_e('Selecciona tipo de transacción','hp-crear-propiedad'); ?></option>
                            <?php
                            $transacciones = get_terms(array(
                                'taxonomy' => 'transaccion',
                                'hide_empty' => false,
                            ));
                            foreach($transacciones as $transaccion):
                            ?>
                                <option value="<?php echo esc_attr($transaccion->term_id); ?>"><?php echo esc_html($transaccion->name); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <div class="hpcp__grid">
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Tipo de propiedad','hp-crear-propiedad'); ?> *</label>
                        <select class="hpcp__select" name="property-type" required>
                            <option value=""><?php esc_html_e('Selecciona tipo de propiedad','hp-crear-propiedad'); ?></option>
                            <?php
                            $property_types = get_terms(array(
                                'taxonomy' => 'property-type',
                                'hide_empty' => false,
                            ));
                            foreach($property_types as $type):
                            ?>
                                <option value="<?php echo esc_attr($type->term_id); ?>"><?php echo esc_html($type->name); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Precio','hp-crear-propiedad'); ?> *</label>
                        <input class="hpcp__input" type="number" min="0" step="0.01" name="precio" required />
                    </div>
                </div>

                <div class="hpcp__grid">
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Tamaño de construcción (m2)','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="number" min="0" step="0.01" name="tamano_de_construccion" />
                    </div>
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Metros de terreno (m2)','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="number" min="0" step="0.01" name="metros_terreno" />
                    </div>
                </div>

                <div class="hpcp__grid">
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Comisión','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="number" min="0" step="0.01" name="comision" />
                    </div>
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Condiciones para comisión','hp-crear-propiedad'); ?></label>
                        <textarea class="hpcp__textarea" name="condiciones_para_comision"></textarea>
                    </div>
                </div>

                <div class="hpcp__grid">
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Habitaciones','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="number" min="0" step="1" name="habitaciones" />
                    </div>
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Baños','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="number" min="0" step="1" name="banos" />
                    </div>
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Número de estacionamientos','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="number" min="0" step="1" name="numero_estacionamientos" />
                    </div>
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Año de construcción','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="text" name="ano_construccion" />
                    </div>
                </div>

                <fieldset class="hpcp__fieldset">
                    <legend class="hpcp__legend"><?php esc_html_e('Amenidades','hp-crear-propiedad'); ?></legend>
                    <?php
                    // Obtener amenidades desde la configuración del admin
                    $available_amenities = class_exists( 'HPCP_Admin' ) ? 
                        HPCP_Admin::get_available_amenities() : 
                        [
                            'Wi-Fi',
                            'Estacionamiento',
                            'Gimnasio',
                            'Petfriendly',
                            'Amueblado'
                        ];
                    
                    foreach ( $available_amenities as $amenity ) :
                    ?>
                        <label class="hpcp__check">
                            <input type="checkbox" name="_amenities[]" value="<?php echo esc_attr($amenity); ?>" />
                            <span><?php echo esc_html($amenity); ?></span>
                        </label>
                    <?php endforeach; ?>
                </fieldset>

                <fieldset class="hpcp__fieldset">
                    <legend class="hpcp__legend"><?php esc_html_e('Ubicación','hp-crear-propiedad'); ?></legend>
                    
                    <?php 
                    $has_maps_api = class_exists( 'HPCP_Admin' ) && ! empty( HPCP_Admin::get_google_maps_api_key() );
                    if ( $has_maps_api ) : 
                    ?>
                        <div class="hpcp__field">
                            <label class="hpcp__label"><?php esc_html_e('Buscar ubicación','hp-crear-propiedad'); ?> *</label>
                            <input class="hpcp__input" type="text" id="location_search" placeholder="<?php esc_attr_e('Busca una dirección, colonia, ciudad...','hp-crear-propiedad'); ?>" />
                            <small class="hpcp__help"><?php esc_html_e('Escribe para buscar en el mapa y autocompletar los campos','hp-crear-propiedad'); ?></small>
                        </div>
                        
                        <div class="hpcp__map-container">
                            <div id="property_map" class="hpcp__map"></div>
                        </div>
                    <?php else : ?>
                        <div class="notice notice-warning inline">
                            <p><?php esc_html_e('⚠️ Google Maps no configurado. Los campos deben llenarse manualmente.','hp-crear-propiedad'); ?></p>
                            <p><small><?php esc_html_e('Administrador: ve a Configuración → Crear Propiedad para configurar Google Maps API.','hp-crear-propiedad'); ?></small></p>
                        </div>
                    <?php endif; ?>
                    
                    <div class="hpcp__grid">
                        <div class="hpcp__field">
                            <label class="hpcp__label"><?php esc_html_e('Estado','hp-crear-propiedad'); ?> *</label>
                            <input class="hpcp__input" type="text" name="estados" id="estados" required <?php echo $has_maps_api ? 'readonly' : 'placeholder="' . esc_attr__('Ej: Ciudad de México, Jalisco','hp-crear-propiedad') . '"'; ?> />
                        </div>
                        <div class="hpcp__field">
                            <label class="hpcp__label"><?php esc_html_e('Ciudad','hp-crear-propiedad'); ?> *</label>
                            <input class="hpcp__input" type="text" name="ciudad" id="ciudad" required <?php echo $has_maps_api ? 'readonly' : 'placeholder="' . esc_attr__('Ej: Guadalajara, Monterrey','hp-crear-propiedad') . '"'; ?> />
                        </div>
                    </div>
                    
                    <div class="hpcp__grid">
                        <div class="hpcp__field">
                            <label class="hpcp__label"><?php esc_html_e('Colonia','hp-crear-propiedad'); ?></label>
                            <input class="hpcp__input" type="text" name="colonia" id="colonia" <?php echo $has_maps_api ? 'readonly' : 'placeholder="' . esc_attr__('Ej: Roma Norte, Polanco','hp-crear-propiedad') . '"'; ?> />
                        </div>
                        <div class="hpcp__field">
                            <label class="hpcp__label"><?php esc_html_e('Código Postal','hp-crear-propiedad'); ?></label>
                            <input class="hpcp__input" type="text" name="codigo_postal" id="codigo_postal" <?php echo $has_maps_api ? 'readonly' : 'placeholder="' . esc_attr__('Ej: 06700','hp-crear-propiedad') . '"'; ?> />
                        </div>
                    </div>
                    
                    <div class="hpcp__grid">
                        <div class="hpcp__field">
                            <label class="hpcp__label">Latitud</label>
                            <input class="hpcp__input" type="text" name="latitud" id="latitud" <?php echo $has_maps_api ? 'readonly' : 'placeholder="19.4326"'; ?> />
                        </div>
                        <div class="hpcp__field">
                            <label class="hpcp__label">Longitud</label>
                            <input class="hpcp__input" type="text" name="longitud" id="longitud" <?php echo $has_maps_api ? 'readonly' : 'placeholder="-99.1332"'; ?> />
                        </div>
                    </div>
                    
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Dirección específica','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="text" name="location" id="location_specific" placeholder="<?php esc_attr_e('Calle y número, referencias, ajustes manuales...','hp-crear-propiedad'); ?>" />
                        <?php if ( $has_maps_api ) : ?>
                            <small class="hpcp__help"><?php esc_html_e('Autocompletado por el mapa, pero puedes editarlo manualmente','hp-crear-propiedad'); ?></small>
                        <?php endif; ?>
                    </div>
                </fieldset>

                <div class="hpcp__actions">
                    <button type="button" class="hpcp__btn hpcp__btn--next"><?php esc_html_e('Siguiente','hp-crear-propiedad'); ?></button>
                </div>
            </section>

            <!-- Step 2: Detalles de la propiedad -->
            <section class="hpcp__step" data-step="2">
                <div class="hpcp__grid">
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Techado (m2)','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="number" step="0.01" name="techado" />
                    </div>
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Patio de maniobras m2','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="number" step="0.01" name="patio_maniobras" />
                    </div>
                </div>

                <div class="hpcp__grid">
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Frente (m)','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="number" step="0.01" name="frente" />
                    </div>
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Fondo (m)','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="number" step="0.01" name="fondo" />
                    </div>
                </div>

                <div class="hpcp__field">
                    <label class="hpcp__label"><?php esc_html_e('Uso de suelo','hp-crear-propiedad'); ?></label>
                    <input class="hpcp__input" type="text" name="uso_de_suelo" />
                </div>

                <div class="hpcp__actions">
                    <button type="button" class="hpcp__btn hpcp__btn--prev"><?php esc_html_e('Anterior','hp-crear-propiedad'); ?></button>
                    <button type="button" class="hpcp__btn hpcp__btn--next"><?php esc_html_e('Siguiente','hp-crear-propiedad'); ?></button>
                </div>
            </section>

            <!-- Step 3: Galería -->
            <section class="hpcp__step" data-step="3">
                <div class="hpcp__field">
                    <label class="hpcp__label"><?php esc_html_e('Imagen destacada','hp-crear-propiedad'); ?> *</label>
                    <input class="hpcp__input" type="file" name="featured_image" accept="image/*" required id="featured_input" />
                    <small class="hpcp__help"><?php esc_html_e('Esta será la imagen principal de la propiedad. Tamaño máximo: 2MB','hp-crear-propiedad'); ?></small>
                    <div id="featured_preview" class="hpcp__featured-preview"></div>
                </div>

                <div class="hpcp__field">
                    <label class="hpcp__label"><?php esc_html_e('Visita virtual (YouTube URL) [_video-tour-url / visita_virtual]','hp-crear-propiedad'); ?></label>
                    <input class="hpcp__input" type="url" name="visita_virtual" placeholder="https://www.youtube.com/watch?v=XXXXXXXX" />
                    <small class="hpcp__help"><?php esc_html_e('Debe ser un enlace de YouTube. Se validará al enviar.','hp-crear-propiedad'); ?></small>
                </div>

                <div class="hpcp__grid">
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Número de pisos','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="number" min="0" step="1" name="pisos" />
                    </div>
                    <div class="hpcp__field">
                        <label class="hpcp__label"><?php esc_html_e('Plantas (imagen)','hp-crear-propiedad'); ?></label>
                        <input class="hpcp__input" type="file" name="plantas" accept="image/*" />
                    </div>
                </div>

                <div class="hpcp__field">
                    <label class="hpcp__label"><?php esc_html_e('Galería de la propiedad (máximo 5 imágenes)','hp-crear-propiedad'); ?></label>
                    <input class="hpcp__input" type="file" name="galeria_propiedad[]" accept="image/*" multiple id="galeria_input" />
                    <small class="hpcp__help"><?php esc_html_e('Selecciona hasta 5 imágenes. Tamaño máximo por imagen: 2MB','hp-crear-propiedad'); ?></small>
                    <div id="galeria_preview" class="hpcp__gallery-preview"></div>
                </div>

                <div class="hpcp__actions">
                    <button type="button" class="hpcp__btn hpcp__btn--prev"><?php esc_html_e('Anterior','hp-crear-propiedad'); ?></button>
                    <button type="button" class="hpcp__btn hpcp__btn--next"><?php esc_html_e('Siguiente','hp-crear-propiedad'); ?></button>
                </div>
            </section>

            <!-- Step 4: Confirmación -->
            <section class="hpcp__step" data-step="4">
                <p class="hpcp__review"><?php esc_html_e('Revisa la información y publica la propiedad. Se publicará inmediatamente.','hp-crear-propiedad'); ?></p>
                <div class="hpcp__actions">
                    <button type="button" class="hpcp__btn hpcp__btn--prev"><?php esc_html_e('Anterior','hp-crear-propiedad'); ?></button>
                    <button type="submit" class="hpcp__btn hpcp__btn--submit"><?php esc_html_e('Publicar propiedad','hp-crear-propiedad'); ?></button>
                </div>
            </section>
        </form>
        <?php
        return ob_get_clean();
    }

    public function render_map_shortcode( $atts ) {
        // Verificar que hay API key
        if ( ! class_exists( 'HPCP_Admin' ) || empty( HPCP_Admin::get_google_maps_api_key() ) ) {
            return '<div class="hpcp-map-notice">' . esc_html__( 'Google Maps no está configurado.', 'hp-crear-propiedad' ) . '</div>';
        }

        // Obtener el post actual
        global $post;
        if ( ! $post || $post->post_type !== 'properties' ) {
            return '<div class="hpcp-map-notice">' . esc_html__( 'Este mapa solo funciona en propiedades.', 'hp-crear-propiedad' ) . '</div>';
        }

        // Obtener datos de ubicación del post
        $latitud = get_post_meta( $post->ID, 'latitud', true );
        $longitud = get_post_meta( $post->ID, 'longitud', true );
        $direccion = get_post_meta( $post->ID, '_address', true );
        $estado = get_post_meta( $post->ID, 'estados', true );
        $ciudad = get_post_meta( $post->ID, 'ciudad', true );
        $colonia = get_post_meta( $post->ID, 'colonia', true );

        // Si no hay coordenadas, no mostrar mapa
        if ( empty( $latitud ) || empty( $longitud ) ) {
            return '<div class="hpcp-map-notice">' . esc_html__( 'No hay información de ubicación para esta propiedad.', 'hp-crear-propiedad' ) . '</div>';
        }

        // Cargar assets
        wp_enqueue_style( 'hpcp-style' );
        if ( wp_script_is( 'google-maps', 'registered' ) ) {
            wp_enqueue_script( 'google-maps' );
        }
        wp_enqueue_script( 'hpcp-display-map', HPCP_URL . 'assets/js/hpcp-display-map.js', ['jquery'], HPCP_VERSION, true );

        // Preparar datos para JavaScript
        $map_data = array(
            'lat' => floatval( $latitud ),
            'lng' => floatval( $longitud ),
            'title' => get_the_title( $post->ID ),
            'direccion' => $direccion,
            'estado' => $estado,
            'ciudad' => $ciudad,
            'colonia' => $colonia,
            'post_id' => $post->ID
        );

        wp_localize_script( 'hpcp-display-map', 'hpcpMapData', $map_data );

        // Atributos del shortcode
        $atts = shortcode_atts( array(
            'height' => '400px',
            'width' => '100%',
            'zoom' => '15',
            'show_info' => 'true'
        ), $atts, 'hp_desplegado_mapa' );

        ob_start(); ?>
        <div class="hpcp-display-map-container" style="width: <?php echo esc_attr($atts['width']); ?>;">
            <?php if ( $atts['show_info'] === 'true' ) : ?>
                <div class="hpcp-location-info">
                    <h4 class="hpcp-location-title"><?php esc_html_e('Ubicación de la propiedad', 'hp-crear-propiedad'); ?></h4>
                    <div class="hpcp-location-details">
                        <?php if ( $direccion ) : ?>
                            <p><strong><?php esc_html_e('Dirección:', 'hp-crear-propiedad'); ?></strong> <?php echo esc_html( $direccion ); ?></p>
                        <?php endif; ?>
                        <?php if ( $colonia ) : ?>
                            <p><strong><?php esc_html_e('Colonia:', 'hp-crear-propiedad'); ?></strong> <?php echo esc_html( $colonia ); ?></p>
                        <?php endif; ?>
                        <?php if ( $ciudad || $estado ) : ?>
                            <p><strong><?php esc_html_e('Ciudad/Estado:', 'hp-crear-propiedad'); ?></strong> 
                            <?php 
                            $location_parts = array_filter([$ciudad, $estado]);
                            echo esc_html( implode(', ', $location_parts) ); 
                            ?>
                            </p>
                        <?php endif; ?>
                        <p><strong><?php esc_html_e('Coordenadas:', 'hp-crear-propiedad'); ?></strong> <?php echo esc_html( $latitud . ', ' . $longitud ); ?></p>
                    </div>
                </div>
            <?php endif; ?>
            
            <div class="hpcp-display-map" 
                 id="hpcp-display-map-<?php echo esc_attr($post->ID); ?>" 
                 style="height: <?php echo esc_attr($atts['height']); ?>; width: 100%; border-radius: 8px; border: 1px solid #ddd;"
                 data-zoom="<?php echo esc_attr($atts['zoom']); ?>">
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    public function handle_submission() {
        if ( empty($_POST['hpcp_action']) || $_POST['hpcp_action'] !== 'create_property' ) return;
        if ( ! isset($_POST['hpcp_nonce']) || ! wp_verify_nonce( $_POST['hpcp_nonce'], 'hpcp_create_property' ) ) return;
        if ( ! is_user_logged_in() || ! current_user_can('edit_posts') ) return;

        // Title and content
        $title = isset($_POST['post_title']) ? sanitize_text_field($_POST['post_title']) : '';
        $content = isset($_POST['post_content']) ? wp_kses_post($_POST['post_content']) : '';
        
        if ( empty($title) ) {
            wp_die( __('Falta el título.','hp-crear-propiedad') );
        }
        if ( empty($content) ) {
            wp_die( __('Falta la descripción de la propiedad.','hp-crear-propiedad') );
        }

        // Validate taxonomies
        $transaccion = isset($_POST['transaccion']) ? intval($_POST['transaccion']) : 0;
        $property_type = isset($_POST['property-type']) ? intval($_POST['property-type']) : 0;
        $estado = isset($_POST['estados']) ? sanitize_text_field($_POST['estados']) : '';
        $ciudad = isset($_POST['ciudad']) ? sanitize_text_field($_POST['ciudad']) : '';
        
        if ( !$transaccion || !$property_type || empty($estado) || empty($ciudad) ) {
            wp_die( __('Debes completar tipo de transacción, tipo de propiedad, estado y ciudad.','hp-crear-propiedad') );
        }

        // Create post as published
        $post_id = wp_insert_post([
            'post_type' => 'properties',
            'post_title' => $title,
            'post_content' => $content,
            'post_status' => 'publish', // Publicar automáticamente
        ], true);

        if ( is_wp_error($post_id) ) {
            wp_die( $post_id->get_error_message() );
        }

        // Sanitize helpers
        $num = function($key){ return isset($_POST[$key]) ? floatval($_POST[$key]) : ''; };
        $int = function($key){ return isset($_POST[$key]) ? intval($_POST[$key]) : ''; };
        $txt = function($key){ return isset($_POST[$key]) ? sanitize_text_field($_POST[$key]) : ''; };

        // Meta map (keys per Crocoblock screenshots)
        $meta = [
            '_hot-deals' => $txt('_hot-deals'),
            'precio' => $num('precio'),
            'tamano_de_construccion' => $num('tamano_de_construccion'),
            'metros_terreno' => $num('metros_terreno'),
            'comision' => $num('comision'),
            'condiciones_para_comision' => isset($_POST['condiciones_para_comision']) ? wp_kses_post($_POST['condiciones_para_comision']) : '',
            'habitaciones' => $int('habitaciones'),
            'banos' => $int('banos'),
            'numero_estacionamientos' => $int('numero_estacionamientos'),
            'ano_construccion' => $txt('ano_construccion'),
            'techado' => $num('techado'),
            'patio_maniobras' => $num('patio_maniobras'),
            'frente' => $num('frente'),
            'fondo' => $num('fondo'),
            'uso_de_suelo' => $txt('uso_de_suelo'),
            'latitud' => $txt('latitud'),
            'longitud' => $txt('longitud'),
            'codigo_postal' => $txt('codigo_postal'),
            'visita_virtual' => $txt('visita_virtual'), // _video-tour-url
            'pisos' => $int('pisos'),
            // Ubicaciones como metadatos
            'estados' => $txt('estados'),
            'ciudad' => $txt('ciudad'),
            'colonia' => $txt('colonia'),
            '_address' => $txt('location'), // Usar el campo location como _address
        ];

        // Amenities array
        if ( isset($_POST['_amenities']) && is_array($_POST['_amenities']) ) {
            $clean = array_map('sanitize_text_field', $_POST['_amenities']);
            $meta['_amenities'] = $clean;
        }

        // Validate YouTube URL if provided
        if ( ! empty($meta['visita_virtual']) ) {
            $yt_pattern = '/^(https?:\/\/)?(www\.)?(youtube\.com|youtu\.be)\/(watch\?v=|embed\/|shorts\/)?[\w\-]{6,}/i';
            if ( ! preg_match($yt_pattern, $meta['visita_virtual']) ) {
                wp_delete_post($post_id, true);
                wp_die( __('La URL de visita virtual debe ser de YouTube.','hp-crear-propiedad') );
            }
        }

        // Save meta
        foreach ( $meta as $k => $v ) {
            if ( $v === '' || $v === null ) continue;
            update_post_meta( $post_id, $k, $v );
        }

        // Handle taxonomies (only transaccion and property-type)
        if ( $transaccion ) {
            wp_set_post_terms( $post_id, [$transaccion], 'transaccion' );
        }
        if ( $property_type ) {
            wp_set_post_terms( $post_id, [$property_type], 'property-type' );
        }

        // Handle featured image
        if ( ! empty($_FILES['featured_image']['name']) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/image.php';
            require_once ABSPATH . 'wp-admin/includes/media.php';
            
            $attach_id = media_handle_upload( 'featured_image', $post_id );
            if ( ! is_wp_error($attach_id) ) {
                set_post_thumbnail( $post_id, $attach_id );
            }
        }

        // Handle single image: plantas
        if ( ! empty($_FILES['plantas']['name']) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/image.php';
            require_once ABSPATH . 'wp-admin/includes/media.php';
            $attach_id = media_handle_upload( 'plantas', $post_id );
            if ( ! is_wp_error($attach_id) ) {
                update_post_meta( $post_id, 'plantas', $attach_id );
            }
        }

        // Handle gallery multiple files (max 5 images, max 2MB each)
        if ( ! empty($_FILES['galeria_propiedad']['name'][0]) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            require_once ABSPATH . 'wp-admin/includes/image.php';
            require_once ABSPATH . 'wp-admin/includes/media.php';

            $max_files = 5;
            $max_size = 2 * 1024 * 1024; // 2MB en bytes
            $valid_files = 0;
            $ids = [];
            
            foreach ( $_FILES['galeria_propiedad']['name'] as $i => $name ) {
                if ( empty($name) ) continue;
                
                // Validar número máximo de archivos
                if ( $valid_files >= $max_files ) {
                    break;
                }
                
                // Validar tamaño del archivo
                if ( $_FILES['galeria_propiedad']['size'][$i] > $max_size ) {
                    continue; // Saltar archivos muy grandes
                }
                
                $file = [
                    'name'     => $_FILES['galeria_propiedad']['name'][$i],
                    'type'     => $_FILES['galeria_propiedad']['type'][$i],
                    'tmp_name' => $_FILES['galeria_propiedad']['tmp_name'][$i],
                    'error'    => $_FILES['galeria_propiedad']['error'][$i],
                    'size'     => $_FILES['galeria_propiedad']['size'][$i],
                ];
                $_FILES['hpcp_tmp'] = $file;
                $aid = media_handle_upload( 'hpcp_tmp', $post_id );
                if ( ! is_wp_error($aid) ) {
                    $ids[] = $aid;
                    $valid_files++;
                }
            }
            if ( $ids ) {
                update_post_meta( $post_id, 'galeria_propiedad', $ids );
            }
        }

        // Redirect to the property URL
        $property_url = get_permalink( $post_id );
        if ( $property_url ) {
            wp_safe_redirect( $property_url );
        } else {
            wp_safe_redirect( home_url( '/?property_created=' . $post_id ) );
        }
        exit;
    }
}
