/**
 * Hunter Price CRM - JavaScript (Versión Simplificada)
 * Sistema completo con AJAX para gestión de prospectos
 */

jQuery(document).ready(function($) {
    'use strict';

    console.log('HP_CRM: Iniciando...');

    // Namespace del CRM
    window.HP_CRM = {
        currentProspect: null,
        prospects: [],
        stats: {}
    };

    // Funciones principales
    function loadProspects() {
        console.log('HP_CRM: Cargando prospectos...');
        console.log('HP_CRM: AJAX URL:', hp_crm_ajax.ajax_url);
        console.log('HP_CRM: Nonce:', hp_crm_ajax.nonce);
        
        showLoading();
        
        $.ajax({
            url: hp_crm_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'hp_crm_get_prospects',
                nonce: hp_crm_ajax.nonce
            },
            success: function(response) {
                console.log('HP_CRM: Respuesta recibida:', response);
                if (response.success) {
                    HP_CRM.prospects = response.data.prospects;
                    HP_CRM.stats = response.data.stats;
                    renderProspects();
                    renderStats();
                } else {
                    console.error('HP_CRM: Error en respuesta:', response);
                    showAlert('error', 'Error al cargar prospectos: ' + (response.data || 'Error desconocido'));
                }
            },
            error: function(xhr, status, error) {
                console.error('HP_CRM: Error AJAX:', xhr, status, error);
                console.error('HP_CRM: Response Text:', xhr.responseText);
                showAlert('error', 'Error de conexión al cargar prospectos. Código: ' + xhr.status);
            },
            complete: function() {
                hideLoading();
            }
        });
    }

    function renderProspects() {
        const container = $('.hp-crm__prospects-table tbody');
        container.empty();

        if (!HP_CRM.prospects || HP_CRM.prospects.length === 0) {
            container.append(`
                <tr>
                    <td colspan="5" class="hp-crm__empty">
                        <div class="hp-crm__empty-icon">👥</div>
                        <div class="hp-crm__empty-message">No tienes prospectos aún</div>
                        <button class="hp-crm__button hp-crm__button--primary hp-crm__button--create-prospect">
                            Crear primer prospecto
                        </button>
                    </td>
                </tr>
            `);
            return;
        }

        HP_CRM.prospects.forEach(function(prospect) {
            const statusClass = 'hp-crm__status--' + prospect.status;
            const statusLabel = getStatusLabel(prospect.status);
            
            const row = `
                <tr data-prospect-id="${prospect.id}">
                    <td>
                        <strong>${escapeHtml(prospect.name)}</strong>
                    </td>
                    <td>${escapeHtml(prospect.phone)}</td>
                    <td>${escapeHtml(prospect.email)}</td>
                    <td>
                        <span class="hp-crm__status ${statusClass}">${statusLabel}</span>
                    </td>
                    <td>
                        <div class="hp-crm__actions">
                            <a href="https://wa.me/52${prospect.phone.replace(/[^0-9]/g, '')}" 
                               target="_blank" 
                               class="hp-crm__button hp-crm__button--small hp-crm__button--whatsapp"
                               title="Enviar WhatsApp">
                                📱
                            </a>
                            <a href="mailto:${prospect.email}" 
                               class="hp-crm__button hp-crm__button--small"
                               title="Enviar Email">
                                ✉️
                            </a>
                            <button class="hp-crm__button hp-crm__button--small hp-crm__button--view-prospect" 
                                    data-prospect-id="${prospect.id}"
                                    title="Ver detalles">
                                👁️
                            </button>
                            <button class="hp-crm__button hp-crm__button--small hp-crm__button--edit-prospect" 
                                    data-prospect-id="${prospect.id}"
                                    title="Editar">
                                ✏️
                            </button>
                            <button class="hp-crm__button hp-crm__button--small hp-crm__button--danger hp-crm__button--delete-prospect" 
                                    data-prospect-id="${prospect.id}"
                                    title="Eliminar">
                                🗑️
                            </button>
                        </div>
                    </td>
                </tr>
            `;
            container.append(row);
        });
    }

    function renderStats() {
        if (!HP_CRM.stats) return;

        $('.hp-crm__stats-card--total .hp-crm__stats-card-number').text(HP_CRM.stats.total || 0);
        $('.hp-crm__stats-card--nuevo .hp-crm__stats-card-number').text(HP_CRM.stats.nuevo || 0);
        $('.hp-crm__stats-card--contactado .hp-crm__stats-card-number').text(HP_CRM.stats.contactado || 0);
        $('.hp-crm__stats-card--interesado .hp-crm__stats-card-number').text(HP_CRM.stats.interesado || 0);
        $('.hp-crm__stats-card--seguimiento .hp-crm__stats-card-number').text(HP_CRM.stats.seguimiento || 0);
        $('.hp-crm__stats-card--vendido .hp-crm__stats-card-number').text(HP_CRM.stats.vendido || 0);
    }

    function showModal(modalId) {
        $('#' + modalId + '-modal').addClass('hp-crm__modal--active');
        $('body').css('overflow', 'hidden');
    }

    function closeModal() {
        $('.hp-crm__modal').removeClass('hp-crm__modal--active');
        $('body').css('overflow', '');
        
        // Reset forms
        $('.hp-crm__modal form').each(function() {
            if (this.reset) {
                this.reset();
            }
        });
        
        // Clear hidden fields
        $('.hp-crm__modal input[type="hidden"]').val('');
        $('#selected-property-id').val('');
        $('#property-search-results').empty().hide();
    }

    function showAlert(type, message) {
        const alertClass = 'hp-crm__alert--' + type;
        const alert = $(`
            <div class="hp-crm__alert ${alertClass}">
                ${message}
            </div>
        `);
        
        $('.hp-crm__container').prepend(alert);
        
        setTimeout(function() {
            alert.fadeOut(function() {
                alert.remove();
            });
        }, 5000);
    }

    function showLoading() {
        if (!$('.hp-crm__loading-overlay').length) {
            $('.hp-crm__container').append(`
                <div class="hp-crm__loading-overlay">
                    <div class="hp-crm__loading"></div>
                </div>
            `);
        }
    }

    function hideLoading() {
        $('.hp-crm__loading-overlay').remove();
    }

    function getStatusLabel(status) {
        const labels = {
            'nuevo': 'Nuevo/Lead',
            'contactado': 'Contactado',
            'interesado': 'Interesado',
            'seguimiento': 'En seguimiento',
            'vendido': 'Cerrado/Vendido',
            'descartado': 'Descartado'
        };
        return labels[status] || status;
    }

    function escapeHtml(text) {
        if (!text) return '';
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }

    // Event handlers
    $(document).on('click', '.hp-crm__button--create-prospect', function(e) {
        e.preventDefault();
        showModal('create-prospect');
    });

    $(document).on('click', '.hp-crm__button--edit-prospect', function(e) {
        e.preventDefault();
        const prospectId = $(this).data('prospect-id');
        const prospect = HP_CRM.prospects.find(function(p) { return p.id == prospectId; });
        
        if (prospect) {
            fillProspectForm(prospect);
            showModal('edit-prospect');
        }
    });

    $(document).on('click', '.hp-crm__button--delete-prospect', function(e) {
        e.preventDefault();
        
        if (!confirm('¿Estás seguro de que deseas eliminar este prospecto? Esta acción no se puede deshacer.')) {
            return;
        }

        const prospectId = $(this).data('prospect-id');

        $.ajax({
            url: hp_crm_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'hp_crm_delete_prospect',
                nonce: hp_crm_ajax.nonce,
                prospect_id: prospectId
            },
            success: function(response) {
                if (response.success) {
                    showAlert('success', response.data);
                    loadProspects();
                } else {
                    showAlert('error', response.data);
                }
            },
            error: function() {
                showAlert('error', 'Error de conexión al eliminar prospecto');
            }
        });
    });

    // Modal close events
    $(document).on('click', '.hp-crm__modal-close', function(e) {
        e.preventDefault();
        e.stopPropagation();
        closeModal();
    });

    $(document).on('click', '.hp-crm__modal', function(e) {
        if ($(e.target).hasClass('hp-crm__modal')) {
            closeModal();
        }
    });

    $(document).on('click', '.hp-crm__modal-content', function(e) {
        e.stopPropagation();
    });

    // ESC key to close modal
    $(document).on('keydown', function(e) {
        if (e.keyCode === 27) {
            closeModal();
        }
    });

    // Form handling
    $(document).on('submit', '#hp-crm-prospect-form', function(e) {
        e.preventDefault();
        
        console.log('HP_CRM: Enviando formulario de prospecto...');
        
        const form = $(this);
        const formData = new FormData(this);
        const isEdit = formData.get('prospect_id') ? true : false;
        
        const data = {
            action: isEdit ? 'hp_crm_update_prospect' : 'hp_crm_create_prospect',
            nonce: hp_crm_ajax.nonce,
            name: formData.get('name'),
            phone: formData.get('phone'),
            email: formData.get('email'),
            status: formData.get('status'),
            notes: formData.get('notes')
        };

        if (isEdit) {
            data.prospect_id = formData.get('prospect_id');
        }

        console.log('HP_CRM: Datos a enviar:', data);

        form.find('button[type="submit"]').prop('disabled', true);

        $.ajax({
            url: hp_crm_ajax.ajax_url,
            type: 'POST',
            data: data,
            success: function(response) {
                console.log('HP_CRM: Respuesta del formulario:', response);
                if (response.success) {
                    showAlert('success', response.data.message || 'Operación exitosa');
                    closeModal();
                    loadProspects();
                    form[0].reset();
                } else {
                    showAlert('error', response.data || 'Error desconocido');
                }
            },
            error: function(xhr, status, error) {
                console.error('HP_CRM: Error en formulario:', xhr, status, error);
                console.error('HP_CRM: Response Text:', xhr.responseText);
                showAlert('error', 'Error de conexión. Código: ' + xhr.status);
            },
            complete: function() {
                form.find('button[type="submit"]').prop('disabled', false);
            }
        });
    });

    function fillProspectForm(prospect) {
        $('#prospect-id').val(prospect.id);
        $('#prospect-name').val(prospect.name);
        $('#prospect-phone').val(prospect.phone);
        $('#prospect-email').val(prospect.email);
        $('#prospect-status').val(prospect.status);
        $('#prospect-notes').val(prospect.notes);
    }

    // Tab switching
    $(document).on('click', '.hp-crm__tabs-nav-item', function(e) {
        e.preventDefault();
        const tab = $(this).data('tab');
        
        // Update nav
        $('.hp-crm__tabs-nav-item').removeClass('hp-crm__tabs-nav-item--active');
        $(this).addClass('hp-crm__tabs-nav-item--active');
        
        // Update content
        $('.hp-crm__tabs-pane').removeClass('hp-crm__tabs-pane--active');
        $('.hp-crm__tabs-pane[data-tab="' + tab + '"]').addClass('hp-crm__tabs-pane--active');
    });

    // Search prospects
    $(document).on('input', '.hp-crm__search-input', function() {
        const search = $(this).val().toLowerCase();
        const rows = $('.hp-crm__prospects-table tbody tr');
        
        rows.each(function() {
            const text = $(this).text().toLowerCase();
            $(this).toggle(text.indexOf(search) !== -1);
        });
    });

    // Check URL params
    function checkUrlParams() {
        const urlParams = new URLSearchParams(window.location.search);
        if (urlParams.has('name') || urlParams.has('phone') || urlParams.has('email')) {
            if (window.history && window.history.replaceState) {
                window.history.replaceState({}, document.title, window.location.pathname);
            }
        }
    }

    // Initialize
    if ($('.hp-crm').length) {
        console.log('HP_CRM: Elemento encontrado, iniciando...');
        loadProspects();
        checkUrlParams();
        console.log('HP_CRM: Inicialización completa');
    }
});