<?php
/**
 * Plugin Name: Hunter Price - CRM
 * Description: Sistema CRM completo para gestión de prospectos, citas, llamadas y seguimiento personalizado por usuario.
 * Version: 2.0.0
 * Author: Hunter Price MX
 * License: GPLv2 or later
 * Text Domain: hunter-price-crm
 *
 * @package HunterPriceCRM
 */

// Bloquear acceso directo.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Definir constantes
define( 'HP_CRM_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'HP_CRM_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'HP_CRM_VERSION', '2.0.0' );

if ( ! class_exists( 'HunterPrice_CRM' ) ) {

    final class HunterPrice_CRM {

        /**
         * Constructor.
         */
        public function __construct() {
            $this->load_dependencies();
            $this->init_hooks();
        }

        /**
         * Cargar dependencias
         */
        private function load_dependencies() {
            $required_files = array(
                'includes/class-database.php',
                'includes/class-prospects.php'
            );

            foreach ( $required_files as $file ) {
                $filepath = HP_CRM_PLUGIN_DIR . $file;
                if ( file_exists( $filepath ) ) {
                    require_once $filepath;
                } else {
                    wp_die( 'Error: No se pudo cargar ' . $file );
                }
            }
        }

        /**
         * Inicializar hooks
         */
        private function init_hooks() {
            // Activación del plugin
            register_activation_hook( __FILE__, array( $this, 'activate' ) );
            
            // Shortcode principal
            add_shortcode( 'hp_crm', array( $this, 'shortcode_hp_crm' ) );
            
            // Scripts y estilos
            add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
            
            // Procesar formularios
            add_action( 'init', array( $this, 'process_forms' ) );
        }

        /**
         * Activar plugin - crear tablas
         */
        public function activate() {
            if ( class_exists( 'HP_CRM_Database' ) ) {
                HP_CRM_Database::create_tables();
            }
            
            // Verificar que las tablas se crearon correctamente
            $this->verify_tables();
        }

        /**
         * Verificar que las tablas existan y crearlas si no existen
         */
        public function verify_tables() {
            global $wpdb;
            
            $prospects_table = $wpdb->prefix . 'hp_crm_prospects';
            
            // Verificar si la tabla principal existe
            $table_exists = $wpdb->get_var( $wpdb->prepare( 
                "SHOW TABLES LIKE %s", 
                $prospects_table 
            ) );
            
            if ( $table_exists != $prospects_table ) {
                // Si no existe, forzar creación
                if ( class_exists( 'HP_CRM_Database' ) ) {
                    HP_CRM_Database::create_tables();
                }
            }
        }

        /**
         * Procesar formularios
         */
        public function process_forms() {
            if ( ! is_user_logged_in() ) {
                return;
            }

            // Verificar que las tablas existan
            $this->verify_tables();

            // Procesar creación de prospecto
            if ( isset( $_POST['hp_crm_action'] ) && $_POST['hp_crm_action'] === 'create_prospect' ) {
                $this->handle_create_prospect();
            }

            // Procesar actualización de prospecto
            if ( isset( $_POST['hp_crm_action'] ) && $_POST['hp_crm_action'] === 'update_prospect' ) {
                $this->handle_update_prospect();
            }

            // Procesar eliminación de prospecto
            if ( isset( $_POST['hp_crm_action'] ) && $_POST['hp_crm_action'] === 'delete_prospect' ) {
                $this->handle_delete_prospect();
            }

            // Procesar actualización de estatus
            if ( isset( $_POST['hp_crm_action'] ) && $_POST['hp_crm_action'] === 'update_status' ) {
                $this->handle_update_status();
            }

            // Procesar agregar llamada
            if ( isset( $_POST['hp_crm_action'] ) && $_POST['hp_crm_action'] === 'add_call' ) {
                $this->handle_add_call();
            }

            // Procesar agregar cita
            if ( isset( $_POST['hp_crm_action'] ) && $_POST['hp_crm_action'] === 'add_appointment' ) {
                $this->handle_add_appointment();
            }

            // Procesar agregar nota
            if ( isset( $_POST['hp_crm_action'] ) && $_POST['hp_crm_action'] === 'add_note' ) {
                $this->handle_add_note();
            }

            // Procesar agregar interés
            if ( isset( $_POST['hp_crm_action'] ) && $_POST['hp_crm_action'] === 'add_interest' ) {
                $this->handle_add_interest();
            }
        }

        /**
         * Manejar creación de prospecto
         */
        private function handle_create_prospect() {
            if ( ! wp_verify_nonce( $_POST['hp_crm_nonce'], 'hp_crm_create_prospect' ) ) {
                wp_die( 'Error de seguridad' );
            }

            $data = array(
                'name' => sanitize_text_field( $_POST['name'] ),
                'phone' => sanitize_text_field( $_POST['phone'] ),
                'email' => sanitize_email( $_POST['email'] ),
                'status' => sanitize_text_field( $_POST['status'] ),
                'notes' => sanitize_textarea_field( $_POST['notes'] )
            );

            if ( empty( $data['name'] ) || empty( $data['phone'] ) || empty( $data['email'] ) ) {
                wp_redirect( add_query_arg( 'crm_error', 'missing_fields', wp_get_referer() ) );
                exit;
            }

            $prospect_id = HP_CRM_Prospects::create_prospect( $data );

            if ( $prospect_id ) {
                wp_redirect( add_query_arg( 'crm_success', 'prospect_created', wp_get_referer() ) );
            } else {
                wp_redirect( add_query_arg( 'crm_error', 'create_failed', wp_get_referer() ) );
            }
            exit;
        }

        /**
         * Manejar actualización de prospecto
         */
        private function handle_update_prospect() {
            if ( ! wp_verify_nonce( $_POST['hp_crm_nonce'], 'hp_crm_update_prospect' ) ) {
                wp_die( 'Error de seguridad' );
            }

            $prospect_id = intval( $_POST['prospect_id'] );
            $data = array(
                'name' => sanitize_text_field( $_POST['name'] ),
                'phone' => sanitize_text_field( $_POST['phone'] ),
                'email' => sanitize_email( $_POST['email'] ),
                'status' => sanitize_text_field( $_POST['status'] ),
                'notes' => sanitize_textarea_field( $_POST['notes'] )
            );

            $updated = HP_CRM_Prospects::update_prospect( $prospect_id, $data );

            if ( $updated ) {
                wp_redirect( add_query_arg( 'crm_success', 'prospect_updated', wp_get_referer() ) );
            } else {
                wp_redirect( add_query_arg( 'crm_error', 'update_failed', wp_get_referer() ) );
            }
            exit;
        }

        /**
         * Manejar eliminación de prospecto
         */
        private function handle_delete_prospect() {
            if ( ! wp_verify_nonce( $_POST['hp_crm_nonce'], 'hp_crm_delete_prospect' ) ) {
                wp_die( 'Error de seguridad' );
            }

            try {
                $prospect_id = intval( $_POST['prospect_id'] );
                
                if ( ! $prospect_id ) {
                    wp_redirect( add_query_arg( 'crm_error', 'missing_fields', wp_get_referer() ) );
                    exit;
                }

                $deleted = HP_CRM_Prospects::delete_prospect( $prospect_id );

                if ( $deleted ) {
                    wp_redirect( add_query_arg( 'crm_success', 'prospect_deleted', get_permalink() ) );
                } else {
                    wp_redirect( add_query_arg( 'crm_error', 'delete_failed', wp_get_referer() ) );
                }
            } catch ( Exception $e ) {
                wp_redirect( add_query_arg( 'crm_error', 'delete_failed', wp_get_referer() ) );
            }
            exit;
        }

        /**
         * Manejar actualización de estatus
         */
        private function handle_update_status() {
            if ( ! wp_verify_nonce( $_POST['hp_crm_nonce'], 'hp_crm_update_status' ) ) {
                wp_die( 'Error de seguridad' );
            }

            $prospect_id = intval( $_POST['prospect_id'] );
            $new_status = sanitize_text_field( $_POST['status'] );

            if ( !$prospect_id || empty( $new_status ) ) {
                wp_redirect( add_query_arg( 'crm_error', 'missing_fields', wp_get_referer() ) );
                exit;
            }

            $updated = HP_CRM_Prospects::update_prospect( $prospect_id, array( 'status' => $new_status ) );

            if ( $updated ) {
                wp_redirect( add_query_arg( 'crm_success', 'status_updated', wp_get_referer() ) );
            } else {
                wp_redirect( add_query_arg( 'crm_error', 'update_failed', wp_get_referer() ) );
            }
            exit;
        }

        /**
         * Manejar agregar llamada
         */
        private function handle_add_call() {
            if ( ! wp_verify_nonce( $_POST['hp_crm_nonce'], 'hp_crm_add_call' ) ) {
                wp_die( 'Error de seguridad' );
            }

            $prospect_id = intval( $_POST['prospect_id'] );
            $data = array(
                'call_date' => sanitize_text_field( $_POST['call_date'] ),
                'call_time' => sanitize_text_field( $_POST['call_time'] ),
                'notes' => sanitize_textarea_field( $_POST['notes'] ),
                'result' => sanitize_text_field( $_POST['result'] )
            );

            if ( !$prospect_id ) {
                wp_redirect( add_query_arg( 'crm_error', 'missing_fields', wp_get_referer() ) );
                exit;
            }

            $call_id = HP_CRM_Prospects::add_call( $prospect_id, $data );

            if ( $call_id ) {
                $view_url = add_query_arg( array( 'crm_view' => 'view', 'prospect_id' => $prospect_id ), get_permalink() );
                wp_redirect( add_query_arg( 'crm_success', 'call_added', $view_url ) );
            } else {
                wp_redirect( add_query_arg( 'crm_error', 'add_failed', wp_get_referer() ) );
            }
            exit;
        }

        /**
         * Manejar agregar cita
         */
        private function handle_add_appointment() {
            if ( ! wp_verify_nonce( $_POST['hp_crm_nonce'], 'hp_crm_add_appointment' ) ) {
                wp_die( 'Error de seguridad' );
            }

            $prospect_id = intval( $_POST['prospect_id'] );
            $data = array(
                'title' => sanitize_text_field( $_POST['title'] ),
                'description' => sanitize_textarea_field( $_POST['description'] ),
                'appointment_date' => sanitize_text_field( $_POST['appointment_date'] ),
                'status' => sanitize_text_field( $_POST['status'] ) ?: 'programada'
            );

            if ( !$prospect_id || empty( $data['title'] ) || empty( $data['appointment_date'] ) ) {
                wp_redirect( add_query_arg( 'crm_error', 'missing_fields', wp_get_referer() ) );
                exit;
            }

            $appointment_id = HP_CRM_Prospects::add_appointment( $prospect_id, $data );

            if ( $appointment_id ) {
                $view_url = add_query_arg( array( 'crm_view' => 'view', 'prospect_id' => $prospect_id ), get_permalink() );
                wp_redirect( add_query_arg( 'crm_success', 'appointment_added', $view_url ) );
            } else {
                wp_redirect( add_query_arg( 'crm_error', 'add_failed', wp_get_referer() ) );
            }
            exit;
        }

        /**
         * Manejar agregar nota
         */
        private function handle_add_note() {
            if ( ! wp_verify_nonce( $_POST['hp_crm_nonce'], 'hp_crm_add_note' ) ) {
                wp_die( 'Error de seguridad' );
            }

            $prospect_id = intval( $_POST['prospect_id'] );
            $data = array(
                'event_type' => 'nota',
                'title' => sanitize_text_field( $_POST['title'] ),
                'description' => sanitize_textarea_field( $_POST['description'] ),
                'event_date' => current_time( 'mysql' )
            );

            if ( !$prospect_id || empty( $data['title'] ) ) {
                wp_redirect( add_query_arg( 'crm_error', 'missing_fields', wp_get_referer() ) );
                exit;
            }

            $event_id = HP_CRM_Prospects::add_event( $prospect_id, $data );

            if ( $event_id ) {
                $view_url = add_query_arg( array( 'crm_view' => 'view', 'prospect_id' => $prospect_id ), get_permalink() );
                wp_redirect( add_query_arg( 'crm_success', 'note_added', $view_url ) );
            } else {
                wp_redirect( add_query_arg( 'crm_error', 'add_failed', wp_get_referer() ) );
            }
            exit;
        }

        /**
         * Manejar agregar interés
         */
        private function handle_add_interest() {
            if ( ! wp_verify_nonce( $_POST['hp_crm_nonce'], 'hp_crm_add_interest' ) ) {
                wp_die( 'Error de seguridad' );
            }

            $prospect_id = intval( $_POST['prospect_id'] );
            $property_id = intval( $_POST['property_id'] );
            $notes = sanitize_textarea_field( $_POST['notes'] );

            if ( !$prospect_id || !$property_id ) {
                wp_redirect( add_query_arg( 'crm_error', 'missing_fields', wp_get_referer() ) );
                exit;
            }

            $interest_id = HP_CRM_Prospects::add_interest( $prospect_id, $property_id, $notes );

            if ( $interest_id ) {
                $view_url = add_query_arg( array( 'crm_view' => 'view', 'prospect_id' => $prospect_id ), get_permalink() );
                wp_redirect( add_query_arg( 'crm_success', 'interest_added', $view_url ) );
            } else {
                wp_redirect( add_query_arg( 'crm_error', 'add_failed', wp_get_referer() ) );
            }
            exit;
        }        /**
         * Enqueue scripts y estilos
         */
        public function enqueue_scripts() {
            if ( ! is_user_logged_in() ) {
                return;
            }

            wp_enqueue_style( 'hp-crm-styles', HP_CRM_PLUGIN_URL . 'assets/css/crm-styles.css', array(), HP_CRM_VERSION );
        }

        /**
         * Shortcode [hp_crm]
         */
        public function shortcode_hp_crm( $atts = array(), $content = null ) {
            if ( ! is_user_logged_in() ) {
                return '<div class="hp-crm__login-required">Debes iniciar sesión para acceder al CRM.</div>';
            }

            // Verificar que las tablas existan antes de proceder
            $this->verify_tables();

            $atts = shortcode_atts( array(
                'view' => 'dashboard'
            ), $atts );

            // Obtener la vista actual desde parámetros GET
            $current_view = isset( $_GET['crm_view'] ) ? sanitize_text_field( $_GET['crm_view'] ) : 'dashboard';
            $prospect_id = isset( $_GET['prospect_id'] ) ? intval( $_GET['prospect_id'] ) : 0;

            // Generar contenido según la vista
            ob_start();
            
            // Mostrar mensajes de éxito/error
            $this->display_messages();
            
            // Navegación
            $this->display_navigation( $current_view );
            
            // Contenido principal
            echo '<div class="hp-crm">';
            
            switch ( $current_view ) {
                case 'create':
                    $this->display_create_prospect_form();
                    break;
                case 'edit':
                    $this->display_edit_prospect_form( $prospect_id );
                    break;
                case 'view':
                    $this->display_prospect_details( $prospect_id );
                    break;
                case 'add_call':
                    $this->display_add_call_form( $prospect_id );
                    break;
                case 'add_appointment':
                    $this->display_add_appointment_form( $prospect_id );
                    break;
                case 'add_note':
                    $this->display_add_note_form( $prospect_id );
                    break;
                case 'add_interest':
                    $this->display_add_interest_form( $prospect_id );
                    break;
                case 'dashboard':
                default:
                    $this->display_dashboard();
                    break;
            }
            
            echo '</div>';
            
            return ob_get_clean();
        }

        /**
         * Mostrar mensajes de éxito/error
         */
        private function display_messages() {
            if ( isset( $_GET['crm_success'] ) ) {
                $message = '';
                switch ( $_GET['crm_success'] ) {
                    case 'prospect_created':
                        $message = 'Prospecto creado exitosamente';
                        break;
                    case 'prospect_updated':
                        $message = 'Prospecto actualizado exitosamente';
                        break;
                    case 'prospect_deleted':
                        $message = 'Prospecto eliminado exitosamente';
                        break;
                    case 'status_updated':
                        $message = 'Estado actualizado exitosamente';
                        break;
                    case 'call_added':
                        $message = 'Llamada registrada exitosamente';
                        break;
                    case 'appointment_added':
                        $message = 'Cita agendada exitosamente';
                        break;
                    case 'note_added':
                        $message = 'Nota agregada exitosamente';
                        break;
                    case 'interest_added':
                        $message = 'Interés en propiedad registrado exitosamente';
                        break;
                }
                if ( $message ) {
                    echo '<div class="hp-crm__alert hp-crm__alert--success">' . esc_html( $message ) . '</div>';
                }
            }

            if ( isset( $_GET['crm_error'] ) ) {
                $message = '';
                switch ( $_GET['crm_error'] ) {
                    case 'missing_fields':
                        $message = 'Por favor complete todos los campos obligatorios';
                        break;
                    case 'create_failed':
                        $message = 'Error al crear el prospecto';
                        break;
                    case 'update_failed':
                        $message = 'Error al actualizar el prospecto';
                        break;
                    case 'delete_failed':
                        $message = 'Error al eliminar el prospecto';
                        break;
                    case 'add_failed':
                        $message = 'Error al agregar la información';
                        break;
                    case 'prospect_not_found':
                        $message = 'Prospecto no encontrado';
                        break;
                }
                if ( $message ) {
                    echo '<div class="hp-crm__alert hp-crm__alert--error">' . esc_html( $message ) . '</div>';
                }
            }
        }

        /**
         * Mostrar navegación
         */
        private function display_navigation( $current_view ) {
            $base_url = get_permalink();
            echo '<nav class="hp-crm__nav">';
            echo '<a href="' . esc_url( $base_url ) . '" class="hp-crm__nav-link' . ( $current_view === 'dashboard' ? ' hp-crm__nav-link--active' : '' ) . '">Dashboard</a>';
            echo '<a href="' . esc_url( add_query_arg( 'crm_view', 'create', $base_url ) ) . '" class="hp-crm__nav-link' . ( $current_view === 'create' ? ' hp-crm__nav-link--active' : '' ) . '">Nuevo Prospecto</a>';
            echo '</nav>';
        }

        /**
         * Mostrar dashboard
         */
        private function display_dashboard() {
            $prospects = HP_CRM_Prospects::get_user_prospects();
            $stats = HP_CRM_Prospects::get_user_stats();
            
            // Estadísticas
            echo '<div class="hp-crm__stats">';
            echo '<div class="hp-crm__stats-card">';
            echo '<h3 class="hp-crm__stats-title">Total Prospectos</h3>';
            echo '<div class="hp-crm__stats-number">' . ($stats['total'] ?? 0) . '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__stats-card">';
            echo '<h3 class="hp-crm__stats-title">Nuevos</h3>';
            echo '<div class="hp-crm__stats-number">' . ($stats['nuevo'] ?? 0) . '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__stats-card">';
            echo '<h3 class="hp-crm__stats-title">En Seguimiento</h3>';
            echo '<div class="hp-crm__stats-number">' . ($stats['seguimiento'] ?? 0) . '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__stats-card">';
            echo '<h3 class="hp-crm__stats-title">Convertidos</h3>';
            echo '<div class="hp-crm__stats-number">' . ($stats['convertido'] ?? 0) . '</div>';
            echo '</div>';
            echo '</div>';
            
            // Lista de prospectos
            echo '<div class="hp-crm__section">';
            echo '<h2 class="hp-crm__section-title">Lista de Prospectos</h2>';
            
            if ( !empty( $prospects ) ) {
                echo '<div class="hp-crm__table-container">';
                echo '<table class="hp-crm__table">';
                echo '<thead>';
                echo '<tr>';
                echo '<th>Nombre</th>';
                echo '<th>Teléfono</th>';
                echo '<th>Email</th>';
                echo '<th>Estado</th>';
                echo '<th>Acciones</th>';
                echo '</tr>';
                echo '</thead>';
                echo '<tbody>';
                
                foreach ( $prospects as $prospect ) {
                    echo '<tr>';
                    echo '<td>' . esc_html( $prospect->name ) . '</td>';
                    echo '<td>' . esc_html( $prospect->phone ) . '</td>';
                    echo '<td>' . esc_html( $prospect->email ) . '</td>';
                    echo '<td><span class="hp-crm__status hp-crm__status--' . esc_attr( $prospect->status ) . '">' . esc_html( ucfirst( $prospect->status ) ) . '</span></td>';
                    echo '<td class="hp-crm__actions">';
                    
                    $view_url = add_query_arg( array( 'crm_view' => 'view', 'prospect_id' => $prospect->id ), get_permalink() );
                    $edit_url = add_query_arg( array( 'crm_view' => 'edit', 'prospect_id' => $prospect->id ), get_permalink() );
                    
                    echo '<a href="' . esc_url( $view_url ) . '" class="hp-crm__btn hp-crm__btn--small hp-crm__btn--secondary">Ver</a> ';
                    echo '<a href="' . esc_url( $edit_url ) . '" class="hp-crm__btn hp-crm__btn--small hp-crm__btn--primary">Editar</a> ';
                    echo '<form style="display:inline-block" method="post" onsubmit="return confirm(\'¿Estás seguro de eliminar este prospecto?\');">';
                    echo '<input type="hidden" name="hp_crm_action" value="delete_prospect">';
                    echo '<input type="hidden" name="prospect_id" value="' . esc_attr( $prospect->id ) . '">';
                    echo wp_nonce_field( 'hp_crm_delete_prospect', 'hp_crm_nonce', true, false );
                    echo '<button type="submit" class="hp-crm__btn hp-crm__btn--small hp-crm__btn--danger">Eliminar</button>';
                    echo '</form>';
                    
                    echo '</td>';
                    echo '</tr>';
                }
                
                echo '</tbody>';
                echo '</table>';
                echo '</div>';
            } else {
                echo '<p>No hay prospectos registrados. <a href="' . esc_url( add_query_arg( 'crm_view', 'create', get_permalink() ) ) . '">Crear el primero</a></p>';
            }
            
            echo '</div>';
        }

        /**
         * Mostrar formulario de crear prospecto
         */
        private function display_create_prospect_form() {
            echo '<div class="hp-crm__section">';
            echo '<h2 class="hp-crm__section-title">Crear Nuevo Prospecto</h2>';
            echo '<form method="post" class="hp-crm__form">';
            echo '<input type="hidden" name="hp_crm_action" value="create_prospect">';
            echo wp_nonce_field( 'hp_crm_create_prospect', 'hp_crm_nonce', true, false );
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="name">Nombre *</label>';
            echo '<input type="text" id="name" name="name" class="hp-crm__form-input" required>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="phone">Teléfono *</label>';
            echo '<input type="tel" id="phone" name="phone" class="hp-crm__form-input" required>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="email">Email *</label>';
            echo '<input type="email" id="email" name="email" class="hp-crm__form-input" required>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="status">Estado</label>';
            echo '<select id="status" name="status" class="hp-crm__form-select">';
            echo '<option value="nuevo">Nuevo</option>';
            echo '<option value="contactado">Contactado</option>';
            echo '<option value="seguimiento">En Seguimiento</option>';
            echo '<option value="convertido">Convertido</option>';
            echo '<option value="perdido">Perdido</option>';
            echo '</select>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="notes">Notas</label>';
            echo '<textarea id="notes" name="notes" class="hp-crm__form-textarea" rows="4"></textarea>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-actions">';
            echo '<button type="submit" class="hp-crm__btn hp-crm__btn--primary">Crear Prospecto</button>';
            echo '<a href="' . esc_url( get_permalink() ) . '" class="hp-crm__btn hp-crm__btn--secondary">Cancelar</a>';
            echo '</div>';
            
            echo '</form>';
            echo '</div>';
        }

        /**
         * Mostrar formulario de editar prospecto
         */
        private function display_edit_prospect_form( $prospect_id ) {
            if ( !$prospect_id ) {
                echo '<div class="hp-crm__alert hp-crm__alert--error">ID de prospecto no válido</div>';
                return;
            }
            
            $prospect = HP_CRM_Prospects::get_prospect( $prospect_id );
            
            if ( !$prospect ) {
                echo '<div class="hp-crm__alert hp-crm__alert--error">Prospecto no encontrado</div>';
                return;
            }
            
            echo '<div class="hp-crm__section">';
            echo '<h2 class="hp-crm__section-title">Editar Prospecto</h2>';
            echo '<form method="post" class="hp-crm__form">';
            echo '<input type="hidden" name="hp_crm_action" value="update_prospect">';
            echo '<input type="hidden" name="prospect_id" value="' . esc_attr( $prospect->id ) . '">';
            echo wp_nonce_field( 'hp_crm_update_prospect', 'hp_crm_nonce', true, false );
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="name">Nombre *</label>';
            echo '<input type="text" id="name" name="name" class="hp-crm__form-input" value="' . esc_attr( $prospect->name ) . '" required>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="phone">Teléfono *</label>';
            echo '<input type="tel" id="phone" name="phone" class="hp-crm__form-input" value="' . esc_attr( $prospect->phone ) . '" required>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="email">Email *</label>';
            echo '<input type="email" id="email" name="email" class="hp-crm__form-input" value="' . esc_attr( $prospect->email ) . '" required>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="status">Estado</label>';
            echo '<select id="status" name="status" class="hp-crm__form-select">';
            $statuses = array( 'nuevo', 'contactado', 'seguimiento', 'convertido', 'perdido' );
            foreach ( $statuses as $status ) {
                $selected = $prospect->status === $status ? ' selected' : '';
                echo '<option value="' . esc_attr( $status ) . '"' . $selected . '>' . esc_html( ucfirst( $status ) ) . '</option>';
            }
            echo '</select>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="notes">Notas</label>';
            echo '<textarea id="notes" name="notes" class="hp-crm__form-textarea" rows="4">' . esc_textarea( $prospect->notes ) . '</textarea>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-actions">';
            echo '<button type="submit" class="hp-crm__btn hp-crm__btn--primary">Actualizar Prospecto</button>';
            echo '<a href="' . esc_url( get_permalink() ) . '" class="hp-crm__btn hp-crm__btn--secondary">Cancelar</a>';
            echo '</div>';
            
            echo '</form>';
            echo '</div>';
        }

        /**
         * Mostrar detalles del prospecto
         */
        private function display_prospect_details( $prospect_id ) {
            if ( !$prospect_id ) {
                echo '<div class="hp-crm__alert hp-crm__alert--error">ID de prospecto no válido</div>';
                return;
            }
            
            $prospect = HP_CRM_Prospects::get_prospect( $prospect_id );
            
            if ( !$prospect ) {
                echo '<div class="hp-crm__alert hp-crm__alert--error">Prospecto no encontrado</div>';
                return;
            }
            
            // Obtener datos relacionados
            $appointments = HP_CRM_Prospects::get_prospect_appointments( $prospect_id );
            $calls = HP_CRM_Prospects::get_prospect_calls( $prospect_id );
            $interests = HP_CRM_Prospects::get_prospect_interests( $prospect_id );
            $events = HP_CRM_Prospects::get_prospect_events( $prospect_id );
            
            echo '<div class="hp-crm__section">';
            echo '<div class="hp-crm__section-header">';
            echo '<h2 class="hp-crm__section-title">Detalles del Prospecto</h2>';
            echo '<div class="hp-crm__section-actions">';
            $edit_url = add_query_arg( array( 'crm_view' => 'edit', 'prospect_id' => $prospect->id ), get_permalink() );
            echo '<a href="' . esc_url( $edit_url ) . '" class="hp-crm__btn hp-crm__btn--primary">Editar</a>';
            echo '</div>';
            echo '</div>';
            
            // Información básica del prospecto
            echo '<div class="hp-crm__prospect-details">';
            echo '<div class="hp-crm__detail-group">';
            echo '<label class="hp-crm__detail-label">Nombre:</label>';
            echo '<div class="hp-crm__detail-value">' . esc_html( $prospect->name ) . '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__detail-group">';
            echo '<label class="hp-crm__detail-label">Teléfono:</label>';
            echo '<div class="hp-crm__detail-value">';
            echo '<a href="tel:' . esc_attr( $prospect->phone ) . '">' . esc_html( $prospect->phone ) . '</a>';
            echo ' <a href="https://wa.me/' . preg_replace( '/[^0-9]/', '', $prospect->phone ) . '" target="_blank" class="hp-crm__whatsapp-link">WhatsApp</a>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__detail-group">';
            echo '<label class="hp-crm__detail-label">Email:</label>';
            echo '<div class="hp-crm__detail-value">';
            echo '<a href="mailto:' . esc_attr( $prospect->email ) . '">' . esc_html( $prospect->email ) . '</a>';
            echo '</div>';
            echo '</div>';
            
            // Formulario para cambiar estatus
            echo '<div class="hp-crm__detail-group">';
            echo '<label class="hp-crm__detail-label">Estado:</label>';
            echo '<div class="hp-crm__detail-value">';
            echo '<form method="post" style="display: inline-flex; align-items: center; gap: 1rem;">';
            echo '<input type="hidden" name="hp_crm_action" value="update_status">';
            echo '<input type="hidden" name="prospect_id" value="' . esc_attr( $prospect->id ) . '">';
            echo wp_nonce_field( 'hp_crm_update_status', 'hp_crm_nonce', true, false );
            echo '<select name="status" class="hp-crm__form-select" style="width: auto;">';
            $statuses = array( 
                'nuevo' => 'Nuevo',
                'contactado' => 'Contactado', 
                'seguimiento' => 'En Seguimiento', 
                'convertido' => 'Convertido', 
                'perdido' => 'Perdido' 
            );
            foreach ( $statuses as $status_key => $status_label ) {
                $selected = $prospect->status === $status_key ? ' selected' : '';
                echo '<option value="' . esc_attr( $status_key ) . '"' . $selected . '>' . esc_html( $status_label ) . '</option>';
            }
            echo '</select>';
            echo '<button type="submit" class="hp-crm__btn hp-crm__btn--small hp-crm__btn--primary">Actualizar</button>';
            echo '</form>';
            echo '</div>';
            echo '</div>';
            
            if ( !empty( $prospect->notes ) ) {
                echo '<div class="hp-crm__detail-group">';
                echo '<label class="hp-crm__detail-label">Notas:</label>';
                echo '<div class="hp-crm__detail-value">' . nl2br( esc_html( $prospect->notes ) ) . '</div>';
                echo '</div>';
            }
            
            echo '<div class="hp-crm__detail-group">';
            echo '<label class="hp-crm__detail-label">Creado:</label>';
            echo '<div class="hp-crm__detail-value">' . esc_html( date( 'j F Y, g:i a', strtotime( $prospect->created_at ) ) ) . '</div>';
            echo '</div>';
            echo '</div>';
            
            // Sección de acciones rápidas
            echo '<div class="hp-crm__quick-actions">';
            echo '<h3 class="hp-crm__section-title">Acciones Rápidas</h3>';
            echo '<div class="hp-crm__actions-grid">';
            
            $add_call_url = add_query_arg( array( 'crm_view' => 'add_call', 'prospect_id' => $prospect->id ), get_permalink() );
            $add_appointment_url = add_query_arg( array( 'crm_view' => 'add_appointment', 'prospect_id' => $prospect->id ), get_permalink() );
            $add_note_url = add_query_arg( array( 'crm_view' => 'add_note', 'prospect_id' => $prospect->id ), get_permalink() );
            $add_interest_url = add_query_arg( array( 'crm_view' => 'add_interest', 'prospect_id' => $prospect->id ), get_permalink() );
            
            echo '<a href="' . esc_url( $add_call_url ) . '" class="hp-crm__action-card">📞 Registrar Llamada</a>';
            echo '<a href="' . esc_url( $add_appointment_url ) . '" class="hp-crm__action-card">📅 Agendar Cita</a>';
            echo '<a href="' . esc_url( $add_note_url ) . '" class="hp-crm__action-card">📝 Agregar Nota</a>';
            echo '<a href="' . esc_url( $add_interest_url ) . '" class="hp-crm__action-card">🏠 Propiedad de Interés</a>';
            echo '</div>';
            echo '</div>';
            
            // Timeline de actividades
            echo '<div class="hp-crm__timeline">';
            echo '<h3 class="hp-crm__section-title">Timeline de Actividades</h3>';
            
            // Mostrar citas
            if ( !empty( $appointments ) ) {
                echo '<h4 class="hp-crm__subsection-title">📅 Citas</h4>';
                echo '<div class="hp-crm__timeline-items">';
                foreach ( $appointments as $appointment ) {
                    echo '<div class="hp-crm__timeline-item">';
                    echo '<div class="hp-crm__timeline-date">' . esc_html( date( 'j M Y', strtotime( $appointment->appointment_date ) ) ) . '</div>';
                    echo '<div class="hp-crm__timeline-content">';
                    echo '<h5>' . esc_html( $appointment->title ) . '</h5>';
                    if ( !empty( $appointment->description ) ) {
                        echo '<p>' . esc_html( $appointment->description ) . '</p>';
                    }
                    echo '<span class="hp-crm__status hp-crm__status--' . esc_attr( $appointment->status ) . '">' . esc_html( ucfirst( $appointment->status ) ) . '</span>';
                    echo '</div>';
                    echo '</div>';
                }
                echo '</div>';
            }
            
            // Mostrar llamadas
            if ( !empty( $calls ) ) {
                echo '<h4 class="hp-crm__subsection-title">📞 Llamadas</h4>';
                echo '<div class="hp-crm__timeline-items">';
                foreach ( $calls as $call ) {
                    echo '<div class="hp-crm__timeline-item">';
                    echo '<div class="hp-crm__timeline-date">' . esc_html( date( 'j M Y', strtotime( $call->call_date ) ) ) . '</div>';
                    echo '<div class="hp-crm__timeline-content">';
                    if ( !empty( $call->call_time ) ) {
                        echo '<p><strong>Duración:</strong> ' . esc_html( $call->call_time ) . '</p>';
                    }
                    if ( !empty( $call->notes ) ) {
                        echo '<p>' . esc_html( $call->notes ) . '</p>';
                    }
                    if ( !empty( $call->result ) ) {
                        echo '<p><strong>Resultado:</strong> ' . esc_html( $call->result ) . '</p>';
                    }
                    echo '</div>';
                    echo '</div>';
                }
                echo '</div>';
            }
            
            // Mostrar intereses en propiedades
            if ( !empty( $interests ) ) {
                echo '<h4 class="hp-crm__subsection-title">🏠 Propiedades de Interés</h4>';
                echo '<div class="hp-crm__timeline-items">';
                foreach ( $interests as $interest ) {
                    echo '<div class="hp-crm__timeline-item">';
                    echo '<div class="hp-crm__timeline-date">' . esc_html( date( 'j M Y', strtotime( $interest->created_at ) ) ) . '</div>';
                    echo '<div class="hp-crm__timeline-content">';
                    echo '<p><strong>Propiedad ID:</strong> ' . esc_html( $interest->property_id ) . '</p>';
                    if ( !empty( $interest->notes ) ) {
                        echo '<p>' . esc_html( $interest->notes ) . '</p>';
                    }
                    echo '</div>';
                    echo '</div>';
                }
                echo '</div>';
            }
            
            // Mostrar eventos/notas
            if ( !empty( $events ) ) {
                echo '<h4 class="hp-crm__subsection-title">📝 Notas y Eventos</h4>';
                echo '<div class="hp-crm__timeline-items">';
                foreach ( $events as $event ) {
                    echo '<div class="hp-crm__timeline-item">';
                    echo '<div class="hp-crm__timeline-date">' . esc_html( date( 'j M Y', strtotime( $event->event_date ) ) ) . '</div>';
                    echo '<div class="hp-crm__timeline-content">';
                    echo '<h5>' . esc_html( $event->title ) . '</h5>';
                    if ( !empty( $event->description ) ) {
                        echo '<p>' . esc_html( $event->description ) . '</p>';
                    }
                    echo '<span class="hp-crm__event-type">' . esc_html( ucfirst( $event->event_type ) ) . '</span>';
                    echo '</div>';
                    echo '</div>';
                }
                echo '</div>';
            }
            
            if ( empty( $appointments ) && empty( $calls ) && empty( $interests ) && empty( $events ) ) {
                echo '<p>No hay actividades registradas para este prospecto.</p>';
            }
            
            echo '</div>';
            echo '</div>';
        }

        /**
         * Mostrar formulario agregar llamada
         */
        private function display_add_call_form( $prospect_id ) {
            if ( !$prospect_id ) {
                echo '<div class="hp-crm__alert hp-crm__alert--error">ID de prospecto no válido</div>';
                return;
            }
            
            $prospect = HP_CRM_Prospects::get_prospect( $prospect_id );
            if ( !$prospect ) {
                echo '<div class="hp-crm__alert hp-crm__alert--error">Prospecto no encontrado</div>';
                return;
            }

            echo '<div class="hp-crm__section">';
            echo '<h2 class="hp-crm__section-title">Registrar Llamada - ' . esc_html( $prospect->name ) . '</h2>';
            echo '<form method="post" class="hp-crm__form">';
            echo '<input type="hidden" name="hp_crm_action" value="add_call">';
            echo '<input type="hidden" name="prospect_id" value="' . esc_attr( $prospect_id ) . '">';
            echo wp_nonce_field( 'hp_crm_add_call', 'hp_crm_nonce', true, false );
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="call_date">Fecha de Llamada</label>';
            echo '<input type="datetime-local" id="call_date" name="call_date" class="hp-crm__form-input" value="' . esc_attr( date( 'Y-m-d\TH:i' ) ) . '">';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="call_time">Duración</label>';
            echo '<input type="text" id="call_time" name="call_time" class="hp-crm__form-input" placeholder="ej: 15 minutos">';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="result">Resultado</label>';
            echo '<select id="result" name="result" class="hp-crm__form-select">';
            echo '<option value="">Seleccionar...</option>';
            echo '<option value="contactado">Contactado exitosamente</option>';
            echo '<option value="no_contesto">No contestó</option>';
            echo '<option value="ocupado">Ocupado - llamar después</option>';
            echo '<option value="interesado">Mostró interés</option>';
            echo '<option value="no_interesado">No interesado</option>';
            echo '<option value="cita_agendada">Cita agendada</option>';
            echo '</select>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="notes">Notas de la llamada</label>';
            echo '<textarea id="notes" name="notes" class="hp-crm__form-textarea" rows="4" placeholder="Detalles de la conversación..."></textarea>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-actions">';
            echo '<button type="submit" class="hp-crm__btn hp-crm__btn--primary">Registrar Llamada</button>';
            $back_url = add_query_arg( array( 'crm_view' => 'view', 'prospect_id' => $prospect_id ), get_permalink() );
            echo '<a href="' . esc_url( $back_url ) . '" class="hp-crm__btn hp-crm__btn--secondary">Cancelar</a>';
            echo '</div>';
            
            echo '</form>';
            echo '</div>';
        }

        /**
         * Mostrar formulario agregar cita
         */
        private function display_add_appointment_form( $prospect_id ) {
            if ( !$prospect_id ) {
                echo '<div class="hp-crm__alert hp-crm__alert--error">ID de prospecto no válido</div>';
                return;
            }
            
            $prospect = HP_CRM_Prospects::get_prospect( $prospect_id );
            if ( !$prospect ) {
                echo '<div class="hp-crm__alert hp-crm__alert--error">Prospecto no encontrado</div>';
                return;
            }

            echo '<div class="hp-crm__section">';
            echo '<h2 class="hp-crm__section-title">Agendar Cita - ' . esc_html( $prospect->name ) . '</h2>';
            echo '<form method="post" class="hp-crm__form">';
            echo '<input type="hidden" name="hp_crm_action" value="add_appointment">';
            echo '<input type="hidden" name="prospect_id" value="' . esc_attr( $prospect_id ) . '">';
            echo wp_nonce_field( 'hp_crm_add_appointment', 'hp_crm_nonce', true, false );
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="title">Título de la Cita *</label>';
            echo '<input type="text" id="title" name="title" class="hp-crm__form-input" required placeholder="ej: Visita a propiedad">';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="appointment_date">Fecha y Hora *</label>';
            echo '<input type="datetime-local" id="appointment_date" name="appointment_date" class="hp-crm__form-input" required>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="status">Estado</label>';
            echo '<select id="status" name="status" class="hp-crm__form-select">';
            echo '<option value="programada">Programada</option>';
            echo '<option value="confirmada">Confirmada</option>';
            echo '<option value="realizada">Realizada</option>';
            echo '<option value="cancelada">Cancelada</option>';
            echo '</select>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="description">Descripción</label>';
            echo '<textarea id="description" name="description" class="hp-crm__form-textarea" rows="4" placeholder="Detalles de la cita, ubicación, etc..."></textarea>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-actions">';
            echo '<button type="submit" class="hp-crm__btn hp-crm__btn--primary">Agendar Cita</button>';
            $back_url = add_query_arg( array( 'crm_view' => 'view', 'prospect_id' => $prospect_id ), get_permalink() );
            echo '<a href="' . esc_url( $back_url ) . '" class="hp-crm__btn hp-crm__btn--secondary">Cancelar</a>';
            echo '</div>';
            
            echo '</form>';
            echo '</div>';
        }

        /**
         * Mostrar formulario agregar nota
         */
        private function display_add_note_form( $prospect_id ) {
            if ( !$prospect_id ) {
                echo '<div class="hp-crm__alert hp-crm__alert--error">ID de prospecto no válido</div>';
                return;
            }
            
            $prospect = HP_CRM_Prospects::get_prospect( $prospect_id );
            if ( !$prospect ) {
                echo '<div class="hp-crm__alert hp-crm__alert--error">Prospecto no encontrado</div>';
                return;
            }

            echo '<div class="hp-crm__section">';
            echo '<h2 class="hp-crm__section-title">Agregar Nota - ' . esc_html( $prospect->name ) . '</h2>';
            echo '<form method="post" class="hp-crm__form">';
            echo '<input type="hidden" name="hp_crm_action" value="add_note">';
            echo '<input type="hidden" name="prospect_id" value="' . esc_attr( $prospect_id ) . '">';
            echo wp_nonce_field( 'hp_crm_add_note', 'hp_crm_nonce', true, false );
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="title">Título de la Nota *</label>';
            echo '<input type="text" id="title" name="title" class="hp-crm__form-input" required placeholder="ej: Seguimiento telefónico">';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="description">Descripción</label>';
            echo '<textarea id="description" name="description" class="hp-crm__form-textarea" rows="6" placeholder="Detalles de la nota..."></textarea>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-actions">';
            echo '<button type="submit" class="hp-crm__btn hp-crm__btn--primary">Guardar Nota</button>';
            $back_url = add_query_arg( array( 'crm_view' => 'view', 'prospect_id' => $prospect_id ), get_permalink() );
            echo '<a href="' . esc_url( $back_url ) . '" class="hp-crm__btn hp-crm__btn--secondary">Cancelar</a>';
            echo '</div>';
            
            echo '</form>';
            echo '</div>';
        }

        /**
         * Mostrar formulario agregar interés
         */
        private function display_add_interest_form( $prospect_id ) {
            if ( !$prospect_id ) {
                echo '<div class="hp-crm__alert hp-crm__alert--error">ID de prospecto no válido</div>';
                return;
            }
            
            $prospect = HP_CRM_Prospects::get_prospect( $prospect_id );
            if ( !$prospect ) {
                echo '<div class="hp-crm__alert hp-crm__alert--error">Prospecto no encontrado</div>';
                return;
            }

            echo '<div class="hp-crm__section">';
            echo '<h2 class="hp-crm__section-title">Agregar Propiedad de Interés - ' . esc_html( $prospect->name ) . '</h2>';
            echo '<form method="post" class="hp-crm__form">';
            echo '<input type="hidden" name="hp_crm_action" value="add_interest">';
            echo '<input type="hidden" name="prospect_id" value="' . esc_attr( $prospect_id ) . '">';
            echo wp_nonce_field( 'hp_crm_add_interest', 'hp_crm_nonce', true, false );
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="property_id">ID de Propiedad *</label>';
            echo '<input type="number" id="property_id" name="property_id" class="hp-crm__form-input" required placeholder="ej: 123">';
            echo '<small class="hp-crm__form-help">Ingresa el ID de la propiedad de interés</small>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-row">';
            echo '<div class="hp-crm__form-group">';
            echo '<label class="hp-crm__form-label" for="notes">Notas del Interés</label>';
            echo '<textarea id="notes" name="notes" class="hp-crm__form-textarea" rows="4" placeholder="Detalles sobre el interés en esta propiedad..."></textarea>';
            echo '</div>';
            echo '</div>';
            
            echo '<div class="hp-crm__form-actions">';
            echo '<button type="submit" class="hp-crm__btn hp-crm__btn--primary">Registrar Interés</button>';
            $back_url = add_query_arg( array( 'crm_view' => 'view', 'prospect_id' => $prospect_id ), get_permalink() );
            echo '<a href="' . esc_url( $back_url ) . '" class="hp-crm__btn hp-crm__btn--secondary">Cancelar</a>';
            echo '</div>';
            
            echo '</form>';
            echo '</div>';
        }
    }
}

// Inicializar plugin.
function hunterprice_crm_bootstrap() {
    static $instance = null;
    if ( null === $instance ) {
        $instance = new HunterPrice_CRM();
    }
    return $instance;
}
add_action( 'plugins_loaded', 'hunterprice_crm_bootstrap' );

