<?php
/**
 * Clase para panel administrativo
 *
 * @package HunterPriceCRM
 */

// Bloquear acceso directo.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class HP_CRM_Admin {

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
    }

    /**
     * Agregar menú en el admin
     */
    public function add_admin_menu() {
        add_menu_page(
            'Hunter Price CRM',
            'HP CRM',
            'manage_options',
            'hp-crm',
            array( $this, 'admin_page' ),
            'dashicons-groups',
            30
        );

        add_submenu_page(
            'hp-crm',
            'Configuración CRM',
            'Configuración',
            'manage_options',
            'hp-crm-settings',
            array( $this, 'settings_page' )
        );
    }

    /**
     * Página principal del admin
     */
    public function admin_page() {
        global $wpdb;
        
        // Estadísticas generales
        $prospects_table = HP_CRM_Database::get_prospects_table();
        $total_prospects = $wpdb->get_var( "SELECT COUNT(*) FROM $prospects_table" );
        
        $appointments_table = HP_CRM_Database::get_appointments_table();
        $total_appointments = $wpdb->get_var( "SELECT COUNT(*) FROM $appointments_table" );
        
        $calls_table = HP_CRM_Database::get_calls_table();
        $total_calls = $wpdb->get_var( "SELECT COUNT(*) FROM $calls_table" );

        // Estadísticas por usuario
        $users_stats = $wpdb->get_results( "
            SELECT 
                u.ID, 
                u.display_name,
                COUNT(p.id) as prospects_count
            FROM {$wpdb->users} u 
            LEFT JOIN $prospects_table p ON u.ID = p.user_id 
            WHERE u.ID IN (
                SELECT DISTINCT user_id FROM $prospects_table
            )
            GROUP BY u.ID
            ORDER BY prospects_count DESC
        " );

        ?>
        <div class="wrap">
            <h1>Hunter Price CRM - Panel Administrativo</h1>
            
            <div class="hp-crm-admin-stats">
                <div class="hp-crm-admin-stats__grid">
                    <div class="hp-crm-admin-stats__card">
                        <h3>Total Prospectos</h3>
                        <span class="hp-crm-admin-stats__number"><?php echo $total_prospects; ?></span>
                    </div>
                    <div class="hp-crm-admin-stats__card">
                        <h3>Total Citas</h3>
                        <span class="hp-crm-admin-stats__number"><?php echo $total_appointments; ?></span>
                    </div>
                    <div class="hp-crm-admin-stats__card">
                        <h3>Total Llamadas</h3>
                        <span class="hp-crm-admin-stats__number"><?php echo $total_calls; ?></span>
                    </div>
                </div>
            </div>

            <div class="hp-crm-admin-users">
                <h2>Estadísticas por Usuario</h2>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th>Usuario</th>
                            <th>Prospectos</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ( $users_stats ) : ?>
                            <?php foreach ( $users_stats as $user_stat ) : ?>
                                <tr>
                                    <td><?php echo esc_html( $user_stat->display_name ); ?></td>
                                    <td><?php echo $user_stat->prospects_count; ?></td>
                                    <td>
                                        <a href="<?php echo admin_url( 'admin.php?page=hp-crm&user_id=' . $user_stat->ID ); ?>" class="button">Ver detalles</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else : ?>
                            <tr>
                                <td colspan="3">No hay datos disponibles</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <div class="hp-crm-admin-info">
                <h2>Información del Plugin</h2>
                <p><strong>Versión:</strong> <?php echo HP_CRM_VERSION; ?></p>
                <p><strong>Shortcode principal:</strong> <code>[hp_crm]</code></p>
                <p><strong>Shortcode para vista de prospecto:</strong> <code>[hp_crm view="prospect"]</code></p>
                
                <h3>Instrucciones de uso:</h3>
                <ol>
                    <li>Crea una página e inserta el shortcode <code>[hp_crm]</code></li>
                    <li>Solo usuarios logueados pueden acceder al CRM</li>
                    <li>Cada usuario ve únicamente sus prospectos</li>
                    <li>El sistema incluye gestión completa de prospectos, citas, llamadas y eventos</li>
                </ol>
            </div>

            <style>
            .hp-crm-admin-stats {
                margin: 20px 0;
            }
            .hp-crm-admin-stats__grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                gap: 20px;
                margin: 20px 0;
            }
            .hp-crm-admin-stats__card {
                background: #fff;
                padding: 20px;
                border: 1px solid #ddd;
                border-radius: 4px;
                text-align: center;
            }
            .hp-crm-admin-stats__card h3 {
                margin: 0 0 10px 0;
                font-size: 14px;
                color: #666;
            }
            .hp-crm-admin-stats__number {
                display: block;
                font-size: 32px;
                font-weight: bold;
                color: #0073aa;
            }
            .hp-crm-admin-users {
                margin: 30px 0;
            }
            .hp-crm-admin-info {
                margin: 30px 0;
                background: #fff;
                padding: 20px;
                border: 1px solid #ddd;
                border-radius: 4px;
            }
            </style>
        </div>
        <?php
    }

    /**
     * Página de configuración
     */
    public function settings_page() {
        if ( isset( $_POST['submit'] ) ) {
            // Guardar configuraciones si es necesario
            update_option( 'hp_crm_settings', $_POST );
            echo '<div class="notice notice-success"><p>Configuración guardada exitosamente.</p></div>';
        }

        $settings = get_option( 'hp_crm_settings', array() );
        ?>
        <div class="wrap">
            <h1>Configuración HP CRM</h1>
            
            <form method="post" action="">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="default_status">Estatus por defecto</label>
                        </th>
                        <td>
                            <select name="default_status" id="default_status">
                                <?php
                                $statuses = HP_CRM_Prospects::get_available_statuses();
                                foreach ( $statuses as $key => $label ) {
                                    $selected = selected( $settings['default_status'] ?? 'nuevo', $key, false );
                                    echo "<option value='{$key}' {$selected}>{$label}</option>";
                                }
                                ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="records_per_page">Registros por página</label>
                        </th>
                        <td>
                            <input type="number" name="records_per_page" id="records_per_page" value="<?php echo $settings['records_per_page'] ?? 10; ?>" min="5" max="100" />
                        </td>
                    </tr>
                </table>
                
                <?php submit_button(); ?>
            </form>

            <hr>

            <h2>Mantenimiento de Base de Datos</h2>
            <p>Si necesitas recrear las tablas del CRM, usa el siguiente botón:</p>
            <a href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=hp-crm-settings&action=recreate_tables' ), 'hp_crm_recreate' ); ?>" 
               class="button button-secondary"
               onclick="return confirm('¿Estás seguro? Esto eliminará todos los datos del CRM.');">
               Recrear Tablas
            </a>

            <?php
            if ( isset( $_GET['action'] ) && $_GET['action'] === 'recreate_tables' && wp_verify_nonce( $_GET['_wpnonce'], 'hp_crm_recreate' ) ) {
                HP_CRM_Database::create_tables();
                echo '<div class="notice notice-success"><p>Tablas recreadas exitosamente.</p></div>';
            }
            ?>
        </div>
        <?php
    }
}