<?php
/**
 * Manejador de peticiones AJAX
 *
 * @package HunterPriceCRM
 */

// Bloquear acceso directo.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class HP_CRM_Ajax_Handler {

    public function __construct() {
        $this->init_hooks();
    }

    private function init_hooks() {
        // Prospectos - para usuarios logueados
        add_action( 'wp_ajax_hp_crm_get_prospects', array( $this, 'get_prospects' ) );
        add_action( 'wp_ajax_hp_crm_create_prospect', array( $this, 'create_prospect' ) );
        add_action( 'wp_ajax_hp_crm_update_prospect', array( $this, 'update_prospect' ) );
        add_action( 'wp_ajax_hp_crm_delete_prospect', array( $this, 'delete_prospect' ) );
        add_action( 'wp_ajax_hp_crm_get_prospect', array( $this, 'get_prospect' ) );

        // Citas
        add_action( 'wp_ajax_hp_crm_add_appointment', array( $this, 'add_appointment' ) );
        add_action( 'wp_ajax_hp_crm_get_appointments', array( $this, 'get_appointments' ) );

        // Llamadas
        add_action( 'wp_ajax_hp_crm_add_call', array( $this, 'add_call' ) );
        add_action( 'wp_ajax_hp_crm_get_calls', array( $this, 'get_calls' ) );

        // Propiedades de interés
        add_action( 'wp_ajax_hp_crm_add_interest', array( $this, 'add_interest' ) );
        add_action( 'wp_ajax_hp_crm_get_interests', array( $this, 'get_interests' ) );
        add_action( 'wp_ajax_hp_crm_search_properties', array( $this, 'search_properties' ) );

        // Eventos
        add_action( 'wp_ajax_hp_crm_add_event', array( $this, 'add_event' ) );
        add_action( 'wp_ajax_hp_crm_get_events', array( $this, 'get_events' ) );
    }

    /**
     * Verificar nonce y usuario logueado
     */
    private function verify_request() {
        // Verificar usuario logueado
        if ( ! is_user_logged_in() ) {
            wp_send_json_error( 'Usuario no autorizado' );
            return false;
        }

        // Verificar nonce
        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'hp_crm_nonce' ) ) {
            wp_send_json_error( 'Verificación de seguridad falló' );
            return false;
        }

        return true;
    }

    /**
     * Obtener prospectos
     */
    public function get_prospects() {
        if ( ! $this->verify_request() ) {
            return;
        }

        try {
            $prospects = HP_CRM_Prospects::get_user_prospects();
            $stats = HP_CRM_Prospects::get_user_stats();

            wp_send_json_success( array(
                'prospects' => $prospects,
                'stats' => $stats
            ) );
        } catch ( Exception $e ) {
            wp_send_json_error( 'Error al obtener prospectos: ' . $e->getMessage() );
        }
    }

    /**
     * Crear prospecto
     */
    public function create_prospect() {
        if ( ! $this->verify_request() ) {
            return;
        }

        try {
            $data = array(
                'name' => sanitize_text_field( $_POST['name'] ?? '' ),
                'phone' => sanitize_text_field( $_POST['phone'] ?? '' ),
                'email' => sanitize_email( $_POST['email'] ?? '' ),
                'status' => sanitize_text_field( $_POST['status'] ?? 'nuevo' ),
                'notes' => sanitize_textarea_field( $_POST['notes'] ?? '' )
            );

            // Validaciones
            if ( empty( $data['name'] ) || empty( $data['phone'] ) || empty( $data['email'] ) ) {
                wp_send_json_error( 'Nombre, teléfono y correo son obligatorios' );
                return;
            }

            $prospect_id = HP_CRM_Prospects::create_prospect( $data );

            if ( $prospect_id ) {
                $prospect = HP_CRM_Prospects::get_prospect( $prospect_id );
                wp_send_json_success( array(
                    'message' => 'Prospecto creado exitosamente',
                    'prospect' => $prospect
                ) );
            } else {
                wp_send_json_error( 'Error al crear el prospecto' );
            }
        } catch ( Exception $e ) {
            wp_send_json_error( 'Error: ' . $e->getMessage() );
        }
    }

    /**
     * Actualizar prospecto
     */
    public function update_prospect() {
        if ( ! $this->verify_request() ) {
            return;
        }

        try {
            $prospect_id = intval( $_POST['prospect_id'] ?? 0 );
            
            if ( ! $prospect_id ) {
                wp_send_json_error( 'ID de prospecto requerido' );
                return;
            }

            $data = array();
            
            if ( isset( $_POST['name'] ) ) $data['name'] = sanitize_text_field( $_POST['name'] );
            if ( isset( $_POST['phone'] ) ) $data['phone'] = sanitize_text_field( $_POST['phone'] );
            if ( isset( $_POST['email'] ) ) $data['email'] = sanitize_email( $_POST['email'] );
            if ( isset( $_POST['status'] ) ) $data['status'] = sanitize_text_field( $_POST['status'] );
            if ( isset( $_POST['notes'] ) ) $data['notes'] = sanitize_textarea_field( $_POST['notes'] );

            $updated = HP_CRM_Prospects::update_prospect( $prospect_id, $data );

            if ( $updated ) {
                $prospect = HP_CRM_Prospects::get_prospect( $prospect_id );
                wp_send_json_success( array(
                    'message' => 'Prospecto actualizado exitosamente',
                    'prospect' => $prospect
                ) );
            } else {
                wp_send_json_error( 'Error al actualizar el prospecto' );
            }
        } catch ( Exception $e ) {
            wp_send_json_error( 'Error: ' . $e->getMessage() );
        }
    }

    /**
     * Eliminar prospecto
     */
    public function delete_prospect() {
        if ( ! $this->verify_request() ) {
            return;
        }

        try {
            $prospect_id = intval( $_POST['prospect_id'] ?? 0 );
            
            if ( ! $prospect_id ) {
                wp_send_json_error( 'ID de prospecto requerido' );
                return;
            }

            $deleted = HP_CRM_Prospects::delete_prospect( $prospect_id );

            if ( $deleted ) {
                wp_send_json_success( array(
                    'message' => 'Prospecto eliminado exitosamente'
                ) );
            } else {
                wp_send_json_error( 'Error al eliminar el prospecto' );
            }
        } catch ( Exception $e ) {
            wp_send_json_error( 'Error: ' . $e->getMessage() );
        }
    }

    /**
     * Obtener prospecto individual
     */
    public function get_prospect() {
        if ( ! $this->verify_request() ) {
            return;
        }

        try {
            $prospect_id = intval( $_POST['prospect_id'] ?? 0 );
            
            if ( ! $prospect_id ) {
                wp_send_json_error( 'ID de prospecto requerido' );
                return;
            }

            $prospect = HP_CRM_Prospects::get_prospect( $prospect_id );

            if ( $prospect ) {
                $appointments = HP_CRM_Prospects::get_prospect_appointments( $prospect_id );
                $calls = HP_CRM_Prospects::get_prospect_calls( $prospect_id );
                $interests = HP_CRM_Prospects::get_prospect_interests( $prospect_id );
                $events = HP_CRM_Prospects::get_prospect_events( $prospect_id );

                wp_send_json_success( array(
                    'prospect' => $prospect,
                    'appointments' => $appointments,
                    'calls' => $calls,
                    'interests' => $interests,
                    'events' => $events
                ) );
            } else {
                wp_send_json_error( 'Prospecto no encontrado' );
            }
        } catch ( Exception $e ) {
            wp_send_json_error( 'Error: ' . $e->getMessage() );
        }
    }

    /**
     * Agregar cita
     */
    public function add_appointment() {
        $this->verify_request();

        $prospect_id = intval( $_POST['prospect_id'] ?? 0 );
        
        if ( ! $prospect_id ) {
            wp_send_json_error( 'ID de prospecto requerido' );
        }

        $data = array(
            'title' => $_POST['title'] ?? '',
            'description' => $_POST['description'] ?? '',
            'appointment_date' => $_POST['appointment_date'] ?? '',
            'status' => $_POST['status'] ?? 'programada'
        );

        if ( empty( $data['title'] ) || empty( $data['appointment_date'] ) ) {
            wp_send_json_error( 'Título y fecha de cita son obligatorios' );
        }

        $appointment_id = HP_CRM_Prospects::add_appointment( $prospect_id, $data );

        if ( $appointment_id ) {
            wp_send_json_success( 'Cita agregada exitosamente' );
        } else {
            wp_send_json_error( 'Error al agregar la cita' );
        }
    }

    /**
     * Obtener citas
     */
    public function get_appointments() {
        $this->verify_request();

        $prospect_id = intval( $_POST['prospect_id'] ?? 0 );
        
        if ( ! $prospect_id ) {
            wp_send_json_error( 'ID de prospecto requerido' );
        }

        $appointments = HP_CRM_Prospects::get_prospect_appointments( $prospect_id );
        wp_send_json_success( $appointments );
    }

    /**
     * Agregar llamada
     */
    public function add_call() {
        $this->verify_request();

        $prospect_id = intval( $_POST['prospect_id'] ?? 0 );
        
        if ( ! $prospect_id ) {
            wp_send_json_error( 'ID de prospecto requerido' );
        }

        $data = array(
            'call_date' => $_POST['call_date'] ?? current_time( 'mysql' ),
            'call_time' => $_POST['call_time'] ?? '',
            'notes' => $_POST['notes'] ?? '',
            'result' => $_POST['result'] ?? ''
        );

        $call_id = HP_CRM_Prospects::add_call( $prospect_id, $data );

        if ( $call_id ) {
            wp_send_json_success( 'Llamada registrada exitosamente' );
        } else {
            wp_send_json_error( 'Error al registrar la llamada' );
        }
    }

    /**
     * Obtener llamadas
     */
    public function get_calls() {
        $this->verify_request();

        $prospect_id = intval( $_POST['prospect_id'] ?? 0 );
        
        if ( ! $prospect_id ) {
            wp_send_json_error( 'ID de prospecto requerido' );
        }

        $calls = HP_CRM_Prospects::get_prospect_calls( $prospect_id );
        wp_send_json_success( $calls );
    }

    /**
     * Agregar interés en propiedad
     */
    public function add_interest() {
        $this->verify_request();

        $prospect_id = intval( $_POST['prospect_id'] ?? 0 );
        $property_id = intval( $_POST['property_id'] ?? 0 );
        
        if ( ! $prospect_id || ! $property_id ) {
            wp_send_json_error( 'ID de prospecto y propiedad requeridos' );
        }

        $notes = $_POST['notes'] ?? '';

        $interest_id = HP_CRM_Prospects::add_interest( $prospect_id, $property_id, $notes );

        if ( $interest_id ) {
            wp_send_json_success( 'Interés agregado exitosamente' );
        } else {
            wp_send_json_error( 'Error al agregar el interés' );
        }
    }

    /**
     * Obtener intereses
     */
    public function get_interests() {
        $this->verify_request();

        $prospect_id = intval( $_POST['prospect_id'] ?? 0 );
        
        if ( ! $prospect_id ) {
            wp_send_json_error( 'ID de prospecto requerido' );
        }

        $interests = HP_CRM_Prospects::get_prospect_interests( $prospect_id );
        wp_send_json_success( $interests );
    }

    /**
     * Buscar propiedades
     */
    public function search_properties() {
        $this->verify_request();

        $search = sanitize_text_field( $_POST['search'] ?? '' );
        
        if ( empty( $search ) ) {
            wp_send_json_error( 'Término de búsqueda requerido' );
        }

        $args = array(
            'post_type' => 'properties',
            'post_status' => 'publish',
            's' => $search,
            'posts_per_page' => 20
        );

        $properties = get_posts( $args );

        $results = array();
        foreach ( $properties as $property ) {
            $results[] = array(
                'ID' => $property->ID,
                'title' => $property->post_title,
                'excerpt' => get_the_excerpt( $property )
            );
        }

        wp_send_json_success( $results );
    }

    /**
     * Agregar evento
     */
    public function add_event() {
        $this->verify_request();

        $prospect_id = intval( $_POST['prospect_id'] ?? 0 );
        
        if ( ! $prospect_id ) {
            wp_send_json_error( 'ID de prospecto requerido' );
        }

        $data = array(
            'event_type' => $_POST['event_type'] ?? 'nota',
            'title' => $_POST['title'] ?? '',
            'description' => $_POST['description'] ?? '',
            'event_date' => $_POST['event_date'] ?? current_time( 'mysql' )
        );

        if ( empty( $data['title'] ) ) {
            wp_send_json_error( 'Título del evento es obligatorio' );
        }

        $event_id = HP_CRM_Prospects::add_event( $prospect_id, $data );

        if ( $event_id ) {
            wp_send_json_success( 'Evento agregado exitosamente' );
        } else {
            wp_send_json_error( 'Error al agregar el evento' );
        }
    }

    /**
     * Obtener eventos
     */
    public function get_events() {
        $this->verify_request();

        $prospect_id = intval( $_POST['prospect_id'] ?? 0 );
        
        if ( ! $prospect_id ) {
            wp_send_json_error( 'ID de prospecto requerido' );
        }

        $events = HP_CRM_Prospects::get_prospect_events( $prospect_id );
        wp_send_json_success( $events );
    }
}