<?php
/**
 * Clase para manejo de prospectos
 *
 * @package HunterPriceCRM
 */

// Bloquear acceso directo.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class HP_CRM_Prospects {

    /**
     * Obtener todos los prospectos del usuario actual
     */
    public static function get_user_prospects( $user_id = null ) {
        global $wpdb;
        
        if ( ! $user_id ) {
            $user_id = get_current_user_id();
        }

        $table = HP_CRM_Database::get_prospects_table();
        
        $results = $wpdb->get_results( $wpdb->prepare(
            "SELECT * FROM $table WHERE user_id = %d ORDER BY updated_at DESC",
            $user_id
        ) );

        return $results;
    }

    /**
     * Obtener un prospecto específico
     */
    public static function get_prospect( $prospect_id, $user_id = null ) {
        global $wpdb;
        
        if ( ! $user_id ) {
            $user_id = get_current_user_id();
        }

        $table = HP_CRM_Database::get_prospects_table();
        
        return $wpdb->get_row( $wpdb->prepare(
            "SELECT * FROM $table WHERE id = %d AND user_id = %d",
            $prospect_id,
            $user_id
        ) );
    }

    /**
     * Crear nuevo prospecto
     */
    public static function create_prospect( $data ) {
        global $wpdb;

        $user_id = get_current_user_id();
        $table = HP_CRM_Database::get_prospects_table();

        $insert_data = array(
            'user_id' => $user_id,
            'name' => sanitize_text_field( $data['name'] ),
            'phone' => sanitize_text_field( $data['phone'] ),
            'email' => sanitize_email( $data['email'] ),
            'status' => sanitize_text_field( $data['status'] ?? 'nuevo' ),
            'notes' => sanitize_textarea_field( $data['notes'] ?? '' )
        );

        $result = $wpdb->insert( $table, $insert_data );

        if ( $result ) {
            return $wpdb->insert_id;
        }

        return false;
    }

    /**
     * Actualizar prospecto
     */
    public static function update_prospect( $prospect_id, $data ) {
        global $wpdb;

        $user_id = get_current_user_id();
        $table = HP_CRM_Database::get_prospects_table();

        // Verificar que el prospecto pertenece al usuario
        $prospect = self::get_prospect( $prospect_id, $user_id );
        if ( ! $prospect ) {
            return false;
        }

        $update_data = array();
        
        if ( isset( $data['name'] ) ) {
            $update_data['name'] = sanitize_text_field( $data['name'] );
        }
        if ( isset( $data['phone'] ) ) {
            $update_data['phone'] = sanitize_text_field( $data['phone'] );
        }
        if ( isset( $data['email'] ) ) {
            $update_data['email'] = sanitize_email( $data['email'] );
        }
        if ( isset( $data['status'] ) ) {
            $update_data['status'] = sanitize_text_field( $data['status'] );
        }
        if ( isset( $data['notes'] ) ) {
            $update_data['notes'] = sanitize_textarea_field( $data['notes'] );
        }

        if ( empty( $update_data ) ) {
            return false;
        }

        $result = $wpdb->update(
            $table,
            $update_data,
            array( 'id' => $prospect_id, 'user_id' => $user_id )
        );

        return $result !== false;
    }

    /**
     * Eliminar prospecto
     */
    public static function delete_prospect( $prospect_id ) {
        global $wpdb;

        $user_id = get_current_user_id();
        $table = HP_CRM_Database::get_prospects_table();

        // Verificar que el prospecto pertenece al usuario
        $prospect = self::get_prospect( $prospect_id, $user_id );
        if ( ! $prospect ) {
            return false;
        }

        // Eliminar registros relacionados
        self::delete_prospect_data( $prospect_id );

        $result = $wpdb->delete(
            $table,
            array( 'id' => $prospect_id, 'user_id' => $user_id )
        );

        return $result !== false;
    }

    /**
     * Eliminar todos los datos relacionados a un prospecto
     */
    private static function delete_prospect_data( $prospect_id ) {
        global $wpdb;

        // Eliminar citas
        $appointments_table = HP_CRM_Database::get_appointments_table();
        $wpdb->delete( $appointments_table, array( 'prospect_id' => $prospect_id ) );

        // Eliminar llamadas
        $calls_table = HP_CRM_Database::get_calls_table();
        $wpdb->delete( $calls_table, array( 'prospect_id' => $prospect_id ) );

        // Eliminar intereses
        $interests_table = HP_CRM_Database::get_interests_table();
        $wpdb->delete( $interests_table, array( 'prospect_id' => $prospect_id ) );

        // Eliminar eventos
        $events_table = HP_CRM_Database::get_events_table();
        $wpdb->delete( $events_table, array( 'prospect_id' => $prospect_id ) );
    }

    /**
     * Obtener estadísticas del usuario
     */
    public static function get_user_stats( $user_id = null ) {
        global $wpdb;
        
        if ( ! $user_id ) {
            $user_id = get_current_user_id();
        }

        $table = HP_CRM_Database::get_prospects_table();
        
        $stats = array();
        
        // Total de prospectos
        $stats['total'] = $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE user_id = %d",
            $user_id
        ) );

        // Por estatus
        $statuses = array( 'nuevo', 'contactado', 'interesado', 'seguimiento', 'vendido', 'descartado' );
        
        foreach ( $statuses as $status ) {
            $stats[$status] = $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM $table WHERE user_id = %d AND status = %s",
                $user_id,
                $status
            ) );
        }

        return $stats;
    }

    /**
     * Obtener estatus disponibles
     */
    public static function get_available_statuses() {
        return array(
            'nuevo' => 'Nuevo/Lead',
            'contactado' => 'Contactado',
            'interesado' => 'Interesado',
            'seguimiento' => 'En seguimiento',
            'vendido' => 'Cerrado/Vendido',
            'descartado' => 'Descartado'
        );
    }

    /**
     * Agregar cita
     */
    public static function add_appointment( $prospect_id, $data ) {
        global $wpdb;

        $user_id = get_current_user_id();
        
        // Verificar que el prospecto pertenece al usuario
        $prospect = self::get_prospect( $prospect_id, $user_id );
        if ( ! $prospect ) {
            return false;
        }

        $table = HP_CRM_Database::get_appointments_table();

        $insert_data = array(
            'prospect_id' => $prospect_id,
            'user_id' => $user_id,
            'title' => sanitize_text_field( $data['title'] ),
            'description' => sanitize_textarea_field( $data['description'] ?? '' ),
            'appointment_date' => sanitize_text_field( $data['appointment_date'] ),
            'status' => sanitize_text_field( $data['status'] ?? 'programada' )
        );

        $result = $wpdb->insert( $table, $insert_data );

        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Agregar llamada
     */
    public static function add_call( $prospect_id, $data ) {
        global $wpdb;

        $user_id = get_current_user_id();
        
        // Verificar que el prospecto pertenece al usuario
        $prospect = self::get_prospect( $prospect_id, $user_id );
        if ( ! $prospect ) {
            return false;
        }

        $table = HP_CRM_Database::get_calls_table();

        $insert_data = array(
            'prospect_id' => $prospect_id,
            'user_id' => $user_id,
            'call_date' => sanitize_text_field( $data['call_date'] ),
            'call_time' => sanitize_text_field( $data['call_time'] ?? '' ),
            'notes' => sanitize_textarea_field( $data['notes'] ?? '' ),
            'result' => sanitize_text_field( $data['result'] ?? '' )
        );

        $result = $wpdb->insert( $table, $insert_data );

        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Agregar propiedad de interés
     */
    public static function add_interest( $prospect_id, $property_id, $notes = '' ) {
        global $wpdb;

        $user_id = get_current_user_id();
        
        // Verificar que el prospecto pertenece al usuario
        $prospect = self::get_prospect( $prospect_id, $user_id );
        if ( ! $prospect ) {
            return false;
        }

        $table = HP_CRM_Database::get_interests_table();

        $insert_data = array(
            'prospect_id' => $prospect_id,
            'property_id' => intval( $property_id ),
            'notes' => sanitize_textarea_field( $notes )
        );

        $result = $wpdb->insert( $table, $insert_data );

        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Agregar evento/nota
     */
    public static function add_event( $prospect_id, $data ) {
        global $wpdb;

        $user_id = get_current_user_id();
        
        // Verificar que el prospecto pertenece al usuario
        $prospect = self::get_prospect( $prospect_id, $user_id );
        if ( ! $prospect ) {
            return false;
        }

        $table = HP_CRM_Database::get_events_table();

        $insert_data = array(
            'prospect_id' => $prospect_id,
            'user_id' => $user_id,
            'event_type' => sanitize_text_field( $data['event_type'] ?? 'nota' ),
            'title' => sanitize_text_field( $data['title'] ),
            'description' => sanitize_textarea_field( $data['description'] ?? '' ),
            'event_date' => sanitize_text_field( $data['event_date'] ?? current_time( 'mysql' ) )
        );

        $result = $wpdb->insert( $table, $insert_data );

        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Obtener citas de un prospecto
     */
    public static function get_prospect_appointments( $prospect_id ) {
        global $wpdb;

        $user_id = get_current_user_id();
        $table = HP_CRM_Database::get_appointments_table();
        
        return $wpdb->get_results( $wpdb->prepare(
            "SELECT * FROM $table WHERE prospect_id = %d AND user_id = %d ORDER BY appointment_date DESC",
            $prospect_id,
            $user_id
        ) );
    }

    /**
     * Obtener llamadas de un prospecto
     */
    public static function get_prospect_calls( $prospect_id ) {
        global $wpdb;

        $user_id = get_current_user_id();
        $table = HP_CRM_Database::get_calls_table();
        
        return $wpdb->get_results( $wpdb->prepare(
            "SELECT * FROM $table WHERE prospect_id = %d AND user_id = %d ORDER BY call_date DESC",
            $prospect_id,
            $user_id
        ) );
    }

    /**
     * Obtener propiedades de interés de un prospecto
     */
    public static function get_prospect_interests( $prospect_id ) {
        global $wpdb;

        $interests_table = HP_CRM_Database::get_interests_table();
        
        $results = $wpdb->get_results( $wpdb->prepare(
            "SELECT i.*, p.post_title as property_name 
             FROM $interests_table i 
             LEFT JOIN {$wpdb->posts} p ON i.property_id = p.ID 
             WHERE i.prospect_id = %d 
             ORDER BY i.created_at DESC",
            $prospect_id
        ) );

        return $results;
    }

    /**
     * Obtener eventos de un prospecto
     */
    public static function get_prospect_events( $prospect_id ) {
        global $wpdb;

        $user_id = get_current_user_id();
        $table = HP_CRM_Database::get_events_table();
        
        return $wpdb->get_results( $wpdb->prepare(
            "SELECT * FROM $table WHERE prospect_id = %d AND user_id = %d ORDER BY event_date DESC",
            $prospect_id,
            $user_id
        ) );
    }
}