// Nota: Cargamos el módulo AJAX dinámicamente con cache-busting para evitar módulos anidados en caché.
// Las funciones (testAjax, getEditableProperty, etc.) se asignarán tras el import dinámico dentro de DOMContentLoaded.

document.addEventListener('DOMContentLoaded', async () => {
  console.log('🚀 DOMContentLoaded - Iniciando admin.js');
  
  // 1) Import dinámico del módulo con anti-caché para evitar que './modules/ajax.js' quede congelado
  const cacheBuster = (window.HPBDWEB_SETTINGS && HPBDWEB_SETTINGS.module_buster) 
    ? HPBDWEB_SETTINGS.module_buster 
    : String(Date.now());
  const modUrl = new URL('./modules/ajax.js', import.meta.url);
  modUrl.searchParams.set('ver', cacheBuster);
  modUrl.searchParams.set('nocache', Math.random().toString().slice(2));

  let ajaxModule;
  try {
    ajaxModule = await import(modUrl.toString());
    console.log('📦 ajax.js exportaciones cargadas:', Object.keys(ajaxModule));
  } catch (e) {
    console.error('❌ Error importando módulo ajax.js:', e);
    return; // No podemos continuar sin el módulo
  }

  // 2) Desestructurar todas las funciones exportadas tal como las usaba el código original
  const {
    testAjax,
    testBdwebApi,
    loadProperties,
    previewProperty,
    getEditableProperty,
    importEditedProperty,
    importProperty,
    importAllBatch,
    loadPropertiesWithPage,
    importPage,
    batchImport,
    debugProperty,
  } = ajaxModule;
  const out = document.getElementById('hpbdweb-output');
  
  if (!out) {
    console.error('❌ No se encontró elemento hpbdweb-output');
    return;
  }
  
  if (!window.HPBDWEB_SETTINGS) {
    console.error('❌ window.HPBDWEB_SETTINGS no disponible');
    return;
  }
  
  console.log('✅ Elementos básicos encontrados, continuando...');

  // Log helper
  const log = (message, type = 'info') => {
    const timestamp = new Date().toLocaleTimeString();
    const prefix = type === 'error' ? '❌' : type === 'success' ? '✅' : 'ℹ️';
    out.textContent += `[${timestamp}] ${prefix} ${message}\n`;
    out.scrollTop = out.scrollHeight;
  };

  // Clear log button
  const clearLogBtn = document.getElementById('hpbdweb-clear-log-btn');
  if (clearLogBtn) {
    clearLogBtn.addEventListener('click', () => {
      out.textContent = 'Log limpiado...\n';
    });
  }

  // Test API button
  const testApiBtn = document.getElementById('hpbdweb-test-api-btn');
  const apiStatus = document.getElementById('hpbdweb-api-status');
  
  console.log('🔍 Elementos encontrados:', {
    apiStatus: !!apiStatus
  });
  
  if (testApiBtn && apiStatus) {
    testApiBtn.addEventListener('click', async (e) => {
      e.preventDefault();
      testApiBtn.disabled = true;
      testApiBtn.textContent = 'Probando...';
      apiStatus.textContent = 'Conectando...';
      apiStatus.className = 'hpbdweb-admin__api-status';

      try {
        const resp = await testBdwebApi();
        log(`API Test: ${resp.data.message}`, 'success');
        log(`Propiedades totales: ${resp.data.properties_found}`);
        if (resp.data.total_pages) {
          log(`Páginas disponibles: ${resp.data.total_pages}`);
        }
        
        const statusText = resp.data.properties_found > 0 
          ? `${resp.data.message} (${resp.data.properties_found} propiedades totales)` 
          : resp.data.message;
        
        apiStatus.textContent = statusText;
        apiStatus.className = 'hpbdweb-admin__api-status hpbdweb-admin__api-status--success';
      } catch (err) {
        const message = err && err.data && err.data.message ? err.data.message : err.message || 'Error desconocido';
        log(`API Error: ${message}`, 'error');
        apiStatus.textContent = `Error: ${message}`;
        apiStatus.className = 'hpbdweb-admin__api-status hpbdweb-admin__api-status--error';
      } finally {
        testApiBtn.disabled = false;
        testApiBtn.textContent = 'Probar API BDweb';
      }
    });
  }

  // Pagination state
  let currentPage = 1;
  let totalPages = 1;
  
  // Load properties button
  const loadPropsBtn = document.getElementById('hpbdweb-load-properties-btn');
  const propsList = document.getElementById('hpbdweb-properties-list');
  const paginationControls = document.getElementById('hpbdweb-pagination-controls');
  
  if (loadPropsBtn && propsList) {
    loadPropsBtn.addEventListener('click', async (e) => {
      e.preventDefault();
      await loadPropertiesPage(1, log);
    });
  }

  // Pagination buttons
  const prevPageBtn = document.getElementById('hpbdweb-prev-page-btn');
  const nextPageBtn = document.getElementById('hpbdweb-next-page-btn');
  const importPageBtn = document.getElementById('hpbdweb-import-page-btn');
  
  if (prevPageBtn) {
    prevPageBtn.addEventListener('click', async (e) => {
      e.preventDefault();
      if (currentPage > 1) {
        await loadPropertiesPage(currentPage - 1, log);
      }
    });
  }
  
  if (nextPageBtn) {
    nextPageBtn.addEventListener('click', async (e) => {
      e.preventDefault();
      if (currentPage < totalPages) {
        await loadPropertiesPage(currentPage + 1, log);
      }
    });
  }

  if (importPageBtn) {
    importPageBtn.addEventListener('click', async (e) => {
      e.preventDefault();
      if (!confirm(`¿Importar todas las propiedades válidas de la página ${currentPage}?`)) {
        return;
      }
      await importCurrentPage(log);
    });
  }

  // Load properties page function
  async function loadPropertiesPage(page, logger) {
    const loadingText = `Cargando página ${page}...`;
    
    loadPropsBtn.disabled = true;
    loadPropsBtn.innerHTML = 'Cargando... <span class="hpbdweb-admin__spinner"></span>';
    propsList.textContent = loadingText;

    try {
      const resp = await loadPropertiesWithPage(page);
      logger(`Propiedades cargadas: ${resp.data.total} (página ${page})`, 'success');
      
      currentPage = page;
      totalPages = resp.data.total_pages || 1;
      
      if (resp.data.properties && resp.data.properties.length > 0) {
        renderPropertiesList(resp.data.properties, propsList, logger);
        
        // Update pagination
        updatePaginationControls();
        
        if (paginationControls) {
          paginationControls.style.display = 'block';
        }
      } else {
        propsList.textContent = 'No se encontraron propiedades.';
        if (paginationControls) {
          paginationControls.style.display = 'none';
        }
      }
    } catch (err) {
      const message = err && err.data && err.data.message ? err.data.message : err.message || 'Error desconocido';
      logger(`Error cargando propiedades: ${message}`, 'error');
      propsList.textContent = `Error: ${message}`;
      if (paginationControls) {
        paginationControls.style.display = 'none';
      }
    } finally {
      loadPropsBtn.disabled = false;
      loadPropsBtn.textContent = 'Cargar Propiedades';
    }
  }

  function updatePaginationControls() {
    const paginationInfo = document.getElementById('hpbdweb-pagination-info');
    const pageStatus = document.getElementById('hpbdweb-page-status');
    
    if (paginationInfo) {
      paginationInfo.textContent = `Página ${currentPage} de ${totalPages}`;
    }
    
    if (prevPageBtn) {
      prevPageBtn.disabled = currentPage <= 1;
    }
    
    if (nextPageBtn) {
      nextPageBtn.disabled = currentPage >= totalPages;
    }
  }

  // Batch import button
  const batchImportBtn = document.getElementById('hpbdweb-batch-import-btn');
  
  if (batchImportBtn) {
    batchImportBtn.addEventListener('click', async (e) => {
      e.preventDefault();
      
      if (!confirm('🔄 IMPORTACIÓN POR LOTES\n\nEsto importará propiedades de a 10, preguntándote cada lote si continuar.\n\n• Más controlado y sin timeouts\n• Puedes pausar cuando quieras\n• Solo importa propiedades con usuarios válidos\n\n¿Iniciar importación por lotes?')) {
        return;
      }
      
      await startBatchImport(log, batchImportBtn);
    });
  }

  // Property preview with editable fields
  const previewBtn = document.getElementById('hpbdweb-preview-btn');
  const oldPreviewBtn = document.getElementById('hpbdweb-old-preview-btn');
  const debugBtn = document.getElementById('hpbdweb-debug-btn');
  const propertyIdInput = document.getElementById('property-id-input');
  const propertyPreview = document.getElementById('hpbdweb-property-preview');
  
  console.log('🔍 Botones de preview encontrados:', {
    previewBtn: !!previewBtn,
    oldPreviewBtn: !!oldPreviewBtn,
    debugBtn: !!debugBtn,
    propertyIdInput: !!propertyIdInput,
    propertyPreview: !!propertyPreview
  });
  
  if (previewBtn && propertyIdInput && propertyPreview) {
    previewBtn.addEventListener('click', async (e) => {
      e.preventDefault();
      const propertyId = propertyIdInput.value.trim();
      
      if (!propertyId) {
        log('ID de propiedad requerido', 'error');
        return;
      }

      previewBtn.disabled = true;
      previewBtn.innerHTML = '📋 Cargando vista previa...';
      propertyPreview.style.display = 'none';

      try {
        console.log('🔍 Obteniendo datos editables para propiedad:', propertyId);
        const resp = await getEditableProperty(propertyId);
        console.log('📋 Respuesta recibida:', resp);
        
        log(`✅ Vista previa editable cargada para propiedad ${propertyId}`, 'success');
        
        if (resp.data.local_user_found) {
          log(`👤 Usuario local encontrado: ${resp.data.author_email}`, 'success');
        } else {
          log(`❌ Usuario local no encontrado para: ${resp.data.author_email}`, 'error');
        }
        
        if (resp.data.already_imported) {
          log(`⚠️ Esta propiedad ya fue importada`, 'error');
        }
        
        console.log('🎨 Renderizando formulario editable...');
        console.log('📄 Contenedor:', propertyPreview);
        console.log('📊 Datos:', resp.data);
        
        renderEditablePropertyForm(resp.data, propertyPreview, log);
        propertyPreview.style.display = 'block';
        
        console.log('✅ Formulario renderizado y mostrado');
      } catch (err) {
        const message = err && err.data && err.data.message ? err.data.message : err.message || 'Error desconocido';
        log(`❌ Error en vista previa: ${message}`, 'error');
        propertyPreview.style.display = 'none';
      } finally {
        previewBtn.disabled = false;
        previewBtn.innerHTML = '📋 Vista Previa Editable';
      }
    });
  }

  // Old preview button (simple preview)
  if (oldPreviewBtn && propertyIdInput && propertyPreview) {
    oldPreviewBtn.addEventListener('click', async (e) => {
      e.preventDefault();
      const propertyId = propertyIdInput.value.trim();
      
      if (!propertyId) {
        log('ID de propiedad requerido', 'error');
        return;
      }

      oldPreviewBtn.disabled = true;
      oldPreviewBtn.textContent = 'Cargando...';
      propertyPreview.style.display = 'none';

      try {
        const resp = await previewProperty(propertyId);
        log(`Vista previa simple cargada para propiedad ${propertyId}`, 'success');
        if (resp.data.local_user_exists) {
          log(`Usuario local encontrado: ${resp.data.local_user_name}`, 'success');
        } else {
          log(`Usuario local no encontrado para: ${resp.data.property.author_email}`, 'error');
        }
        renderPropertyPreview(
          resp.data.property, 
          resp.data.already_imported, 
          propertyPreview, 
          log,
          resp.data.local_user_exists,
          resp.data.local_user_name
        );
        propertyPreview.style.display = 'block';
      } catch (err) {
        const message = err && err.data && err.data.message ? err.data.message : err.message || 'Error desconocido';
        log(`Error en vista previa simple: ${message}`, 'error');
        propertyPreview.style.display = 'none';
      } finally {
        oldPreviewBtn.disabled = false;
        oldPreviewBtn.textContent = '👀 Vista Previa Simple';
      }
    });
  }

  // Debug button
  if (debugBtn && propertyIdInput) {
    debugBtn.addEventListener('click', async (e) => {
      e.preventDefault();
      const propertyId = propertyIdInput.value.trim();
      
      if (!propertyId) {
        log('ID de propiedad requerido para debug', 'error');
        return;
      }

      debugBtn.disabled = true;
      debugBtn.textContent = '🔍 Debugeando...';
      
      try {
        const resp = await debugProperty(propertyId);
        const debug = resp.data.debug_info;
        
        log('🔍 === DEBUG IMPORTACIÓN ===', 'info');
        log(`📋 Propiedad: ${debug.title} (ID: ${debug.property_id})`, 'info');
        log(`📧 Email autor: ${debug.author_info ? debug.author_info.email : debug.author_email}`, 'info');
        
        if (debug.local_user_found) {
          log(`✅ Usuario local encontrado:`, 'success');
          log(`  • Nombre: ${debug.local_user_info.display_name}`, 'success');
          log(`  • Login: ${debug.local_user_info.user_login}`, 'success');
          log(`  • Email: ${debug.local_user_info.user_email}`, 'success');
          log(`  • ID: ${debug.local_user_info.ID}`, 'success');
        } else {
          log(`❌ Usuario local NO encontrado para: ${debug.author_email}`, 'error');
          log(`� Necesitas crear un usuario con este email en WordPress`, 'info');
        }
        
        if (debug.already_imported) {
          log(`⚠️ Propiedad ya fue importada anteriormente`, 'error');
        } else {
          log(`✅ Propiedad no ha sido importada`, 'success');
        }
        
        log(`🏠 Detalles de la propiedad:`, 'info');
        const details = debug.property_details;
        log(`  • Precio: $${details.price}`, 'info');
        log(`  • Habitaciones: ${details.bedrooms}`, 'info');
        log(`  • Baños: ${details.bathrooms}`, 'info');
        log(`  • Tamaño construcción: ${details.property_size}`, 'info');
        log(`  • Tamaño terreno: ${details.area_size}`, 'info');
        log(`  • Año construcción: ${details.year_built}`, 'info');

        log(`📍 Ubicación:`, 'info');
        const location = debug.location_info;
        log(`  • Dirección: ${location.address}`, 'info');
        log(`  • Colonia: ${location.colonia}`, 'info');
        log(`  • Ciudad: ${location.ciudad}`, 'info');
        log(`  • Estado: ${location.estado}`, 'info');
        log(`  • Código Postal: ${location.codigo_postal}`, 'info');
        
        if (location.has_coordinates) {
          log(`  • 🌍 Coordenadas: ${location.latitud}, ${location.longitud}`, 'success');
          log(`  • 🗺️ Google Maps: https://maps.google.com/?q=${location.latitud},${location.longitud}`, 'info');
        } else {
          log(`  • ⚠️ Sin coordenadas GPS`, 'error');
          
          if (location.can_geocode) {
            log(`  • 🚀 Se obtendrán coordenadas automáticamente desde la dirección`, 'success');
          } else if (location.address !== 'Sin dirección') {
            log(`  • ❌ ${location.geocoding_status} - No se pueden obtener coordenadas`, 'error');
            log(`  • 💡 Configura tu Google Maps API Key para geocoding automático`, 'info');
          } else {
            log(`  • ❌ Sin dirección - No se puede hacer geocoding`, 'error');
          }
        }

        log(`📷 Imágenes disponibles:`, 'info');
        const media = debug.media_info || {};
        
        if (media.featured_media_id && media.featured_media_id !== 0) {
          log(`  • ✅ Imagen destacada: ID ${media.featured_media_id}`, 'success');
        } else {
          log(`  • ❌ Sin imagen destacada`, 'error');
        }
        
        if (media.gallery_images_count > 0) {
          log(`  • ✅ Galería: ${media.gallery_images_count} imágenes disponibles`, 'success');
          if (media.gallery_ids && media.gallery_ids.length > 0) {
            log(`  • 📋 Primeras IDs: [${media.gallery_ids.join(', ')}]`, 'info');
          }
        } else {
          log(`  • ❌ Sin galería`, 'error');
        }
        
        log(`${debug.ready_to_import}`, debug.ready_to_import.includes('✅') ? 'success' : 'error');
        
      } catch (err) {
        const message = err && err.data && err.data.message ? err.data.message : err.message || 'Error desconocido';
        log(`Error en debug: ${message}`, 'error');
      } finally {
        debugBtn.disabled = false;
        debugBtn.textContent = '🔍 Debug';
      }
    });
  }

  // Functions
  function renderPropertiesList(properties, container, logger) {
    container.innerHTML = '';
    
    properties.forEach(property => {
      const item = document.createElement('div');
      item.className = 'hpbdweb-admin__property-item';
      
      // Property status indicators
      let statusIndicators = '';
      
      // User status
      if (property.local_user_exists) {
        statusIndicators += `<span class="hpbdweb-admin__user-status hpbdweb-admin__user-status--exists" title="Usuario local: ${property.local_user_name}">✅ ${property.local_user_name}</span>`;
      } else {
        statusIndicators += `<span class="hpbdweb-admin__user-status hpbdweb-admin__user-status--missing" title="No existe usuario local con este email">❌ Sin usuario local</span>`;
      }
      
      // Import status
      if (property.already_imported) {
        statusIndicators += `<br><span class="hpbdweb-admin__import-status hpbdweb-admin__import-status--imported" title="Ya fue importada por este autor">📋 Ya importada</span>`;
      }

      // Import button - disabled if no local user or already imported
      let importButton;
      if (property.already_imported) {
        importButton = `<button class="button" disabled title="Propiedad ya importada">Ya importada</button>`;
      } else if (!property.local_user_exists) {
        importButton = `<button class="button button-primary" disabled title="No se puede importar: usuario no encontrado">Importar</button>`;
      } else {
        importButton = `<button class="button button-primary" onclick="importPropertyFromList(${property.id})">Importar</button>`;
      }
      
      item.innerHTML = `
        <span class="hpbdweb-admin__property-id">ID: ${property.id}</span>
        <div class="hpbdweb-admin__property-info">
          <h4 class="hpbdweb-admin__property-title">${property.title}</h4>
          <div class="hpbdweb-admin__property-meta">
            Autor: ${property.author_name} (${property.author_email})<br>
            ${statusIndicators}<br>
            Fecha: ${new Date(property.date).toLocaleDateString()}
          </div>
        </div>
        <div class="hpbdweb-admin__property-actions">
          <button class="button" onclick="previewPropertyFromList(${property.id})">Vista Previa</button>
          <button class="button button-secondary" onclick="editablePreviewFromList(${property.id})">📋 Editable</button>
          ${importButton}
        </div>
      `;
      
      container.appendChild(item);
    });
  }

  function renderPropertyPreview(property, alreadyImported, container, logger, localUserExists = false, localUserName = null) {
    // User status for preview
    const userStatusHtml = localUserExists 
      ? `<p style="color: #46b450;"><strong>✅ Usuario local encontrado:</strong> ${localUserName}</p>`
      : `<p style="color: #d63638;"><strong>❌ Usuario local no encontrado</strong><br><small>No se puede importar sin un usuario local con email: ${property.author_email}</small></p>`;

    // Import status indicator for preview
    let importStatusHtml = '';
    if (alreadyImported) {
      importStatusHtml = '<p style="color: #0c5460; background: #bee5eb; padding: 0.5rem; border-radius: 0.25rem;"><strong>📋 Esta propiedad ya fue importada por este autor</strong></p>';
    }

    // Import button logic
    let importButtonHtml = '';
    if (alreadyImported) {
      importButtonHtml = '<button class="button" disabled>Ya importada</button>';
    } else if (!localUserExists) {
      importButtonHtml = '<button class="button button-primary" disabled>No se puede importar</button>';
    } else {
      importButtonHtml = `<button class="button button-primary" onclick="importPropertyFromPreview(${property.id})">Importar Propiedad</button>`;
    }

    container.innerHTML = `
      <div class="hpbdweb-admin__preview-title">${property.title}</div>
      <div class="hpbdweb-admin__preview-meta">
        <strong>ID:</strong> ${property.id} | 
        <strong>Autor:</strong> ${property.author_name} (${property.author_email})<br>
        <strong>Precio:</strong> $${property.price} | 
        <strong>Habitaciones:</strong> ${property.bedrooms} | 
        <strong>Baños:</strong> ${property.bathsrooms}
      </div>
      <p>${property.content}</p>
      ${userStatusHtml}
      ${importStatusHtml}
      <div class="hpbdweb-admin__preview-actions">
        ${importButtonHtml}
      </div>
    `;
  }

  // Global functions for inline events
  window.previewPropertyFromList = async (propertyId) => {
    propertyIdInput.value = propertyId;
    previewBtn.click();
  };

  window.editablePreviewFromList = async (propertyId) => {
    propertyIdInput.value = propertyId;
    previewBtn.click(); // Usar el mismo botón que tiene la nueva funcionalidad
  };

  window.importPropertyFromList = async (propertyId) => {
    await importPropertyById(propertyId, log);
  };

  window.importPropertyFromPreview = async (propertyId) => {
    await importPropertyById(propertyId, log);
  };

  async function importPropertyById(propertyId, logger) {
    logger(`Iniciando importación de propiedad ${propertyId}...`);
    
    try {
      const resp = await importProperty(propertyId);
      logger(`${resp.data.message}`, 'success');
      
      if (resp.data.assigned_user) {
        logger(`Asignada a: ${resp.data.assigned_user}`, 'success');
      }
      
      if (resp.data.post_url) {
        logger(`Ver propiedad: ${resp.data.post_url}`);
      }
      
      // Mostrar log de importación de imágenes si está disponible
      if (resp.data.import_log) {
        logger('📷 Log de importación de imágenes:', 'info');
        const logLines = resp.data.import_log.split('\n');
        logLines.forEach(line => {
          if (line.trim()) {
            const type = line.includes('✅') ? 'success' : 
                        line.includes('❌') ? 'error' : 'info';
            logger(`  ${line.trim()}`, type);
          }
        });
      }
      
      // Refresh preview if it's the same property
      if (propertyIdInput && propertyIdInput.value == propertyId) {
        setTimeout(() => previewBtn.click(), 1000);
      }
      
    } catch (err) {
      // Extraer información detallada del error
      let errorMessage = 'Error desconocido';
      let errorDetails = '';
      
      if (err && err.message) {
        errorMessage = err.message;
      } else if (err && err.data && err.data.message) {
        errorMessage = err.data.message;
      }
      
      // Log detallado para el desarrollador
      console.error('Error completo de importación:', {
        propertyId,
        error: err,
        stack: err?.stack,
        name: err?.name
      });
      
      logger(`❌ Error importando propiedad ${propertyId}: ${errorMessage}`, 'error');
      
      // Si hay información adicional, mostrarla
      if (err?.name) {
        logger(`🔍 Tipo de error: ${err.name}`, 'error');
      }
      if (err?.stack) {
        logger(`📋 Stack trace disponible en consola del navegador`, 'info');
      }
    }
  }

  // Import current page
  async function importCurrentPage(logger) {
    importPageBtn.disabled = true;
    importPageBtn.innerHTML = 'Importando... <span class="hpbdweb-admin__spinner"></span>';
    
    logger(`📦 Importando página ${currentPage}...`, 'success');
    
    try {
      const resp = await importPage(currentPage);
      const results = resp.data.results;
      
      // Log details
      results.details.forEach(detail => {
        const type = detail.includes('Error') ? 'error' : 
                    detail.includes('Ya importada') || detail.includes('Sin usuario') ? 'info' : 'success';
        logger(detail, type);
      });
      
      logger(resp.data.message, 'success');
      
      // Reload current page to update status
      setTimeout(() => loadPropertiesPage(currentPage, logger), 1000);
      
    } catch (err) {
      const message = err && err.data && err.data.message ? err.data.message : err.message || 'Error desconocido';
      logger(`❌ Error importando página: ${message}`, 'error');
    } finally {
      importPageBtn.disabled = false;
      importPageBtn.textContent = '📦 Importar Esta Página';
    }
  }

  // Batch import functionality
  async function startBatchImport(logger, button) {
    let currentOffset = 0;
    let totalImported = 0;
    let totalSkipped = 0;
    let totalErrors = 0;
    let continueImport = true;
    
    button.disabled = true;
    button.innerHTML = 'Importando por lotes... <span class="hpbdweb-admin__spinner"></span>';
    
    logger('🔄 Iniciando importación por lotes...', 'success');
    
    try {
      while (continueImport) {
        logger(`📦 Procesando lote desde posición ${currentOffset + 1}...`);
        
        const resp = await batchImport(currentOffset);
        const results = resp.data.results;
        const batchInfo = resp.data.batch_info;
        const pagination = resp.data.pagination;
        
        totalImported += results.imported;
        totalSkipped += results.skipped;
        totalErrors += results.errors;
        
        // Log batch details
        logger(`📊 Lote ${batchInfo.start}-${batchInfo.end} de ${batchInfo.total} procesado:`, 'info');
        results.details.forEach(detail => {
          const type = detail.includes('Error') ? 'error' : 
                      detail.includes('Ya importada') || detail.includes('Sin usuario') ? 'info' : 'success';
          logger(`  ${detail}`, type);
        });
        
        logger(`✅ ${results.imported} importadas, ⏭️ ${results.skipped} omitidas, ❌ ${results.errors} errores`, 'info');
        logger(`📈 Progreso: ${pagination.progress_percent}% (${pagination.next_offset}/${batchInfo.total})`, 'info');
        
        // Check if there are more batches
        if (!pagination.has_more) {
          logger('🎉 ¡Importación completa! No hay más propiedades.', 'success');
          break;
        }
        
        currentOffset = pagination.next_offset;
        
        // Ask user if they want to continue every 10 properties
        const continueMsg = `📊 LOTE COMPLETADO\n\n` +
                          `Procesadas: ${batchInfo.start}-${batchInfo.end} de ${batchInfo.total}\n` +
                          `Progreso: ${pagination.progress_percent}%\n\n` +
                          `Este lote: ${results.imported} importadas, ${results.skipped} omitidas, ${results.errors} errores\n` +
                          `Total hasta ahora: ${totalImported} importadas, ${totalSkipped} omitidas, ${totalErrors} errores\n\n` +
                          `¿Continuar con el siguiente lote de 10 propiedades?`;
        
        continueImport = confirm(continueMsg);
        
        if (!continueImport) {
          logger('⏸️ Importación pausada por el usuario', 'info');
          break;
        }
        
        // Small delay to prevent overwhelming the server
        await new Promise(resolve => setTimeout(resolve, 500));
      }
      
      logger(`🏁 Importación por lotes finalizada:`, 'success');
      logger(`📈 Totales: ${totalImported} importadas, ${totalSkipped} omitidas, ${totalErrors} errores`, 'success');
      
    } catch (err) {
      const message = err && err.data && err.data.message ? err.data.message : err.message || 'Error desconocido';
      logger(`❌ Error en importación por lotes: ${message}`, 'error');
    } finally {
      button.disabled = false;
      button.textContent = '🔄 Importación por Lotes';
    }
  }

  // Bulk import functionality
  async function startBulkImport(logger, button, statusElement) {
    let totalImported = 0;
    let totalSkipped = 0;
    let totalErrors = 0;
    let currentPage = 1;
    
    button.disabled = true;
    button.innerHTML = 'Importando... <span class="hpbdweb-admin__spinner"></span>';
    
    logger('🚀 Iniciando importación masiva...', 'success');
    
    try {
      while (true) {
        logger(`📄 Procesando página ${currentPage}...`);
        
        const resp = await importAllBatch(currentPage);
        const results = resp.data.results;
        
        totalImported += results.imported;
        totalSkipped += results.skipped;
        totalErrors += results.errors;
        
        // Log details
        results.details.forEach(detail => {
          const type = detail.includes('Error') ? 'error' : 
                      detail.includes('Ya importada') || detail.includes('Sin usuario') ? 'info' : 'success';
          logger(detail, type);
        });
        
        logger(resp.data.message, 'info');
        
        // Update status
        if (statusElement) {
          statusElement.textContent = `Total: ${totalImported} importadas, ${totalSkipped} omitidas, ${totalErrors} errores`;
        }
        
        // Check if we should continue
        if (!resp.data.has_more_pages) {
          break;
        }
        
        currentPage = resp.data.next_page;
        
        // Small delay to prevent overwhelming the server
        await new Promise(resolve => setTimeout(resolve, 1000));
      }
      
      logger(`✅ Importación masiva completada: ${totalImported} importadas, ${totalSkipped} omitidas, ${totalErrors} errores`, 'success');
      
    } catch (err) {
      const message = err && err.data && err.data.message ? err.data.message : err.message || 'Error desconocido';
      logger(`❌ Error en importación masiva: ${message}`, 'error');
    } finally {
      button.disabled = false;
      button.textContent = '🚀 Importar Todo';
    }
  }

  // Function to render editable property form
  function renderEditablePropertyForm(propertyData, container, logger) {
    console.log('🎨 renderEditablePropertyForm llamada con:', {
      propertyData,
      container,
      containerExists: !!container
    });

    if (!container) {
      console.error('❌ Container no existe!');
      logger('❌ Error: Contenedor no encontrado', 'error');
      return;
    }

    if (!propertyData) {
      console.error('❌ PropertyData no existe!');
      logger('❌ Error: Datos de propiedad no encontrados', 'error');
      return;
    }
    
    const {
      id, title, content, excerpt, author_name, author_email, author_phone,
      price, comision, condiciones_para_comision,
      bedrooms, bathrooms, property_size, area_size, year_built,
      address, latitude, longitude, codigo_postal, colonia, ciudad, estado,
      video,
      amenidades,
      techado = '', patio_maniobras = '', frente = '', fondo = '', uso_suelo = '',
      already_imported, local_user_found, can_import
    } = propertyData;

    // Asegurar variables para clasificación con retrocompatibilidad
    const txRaw = propertyData.transaccion ?? propertyData.purpose ?? [];
    const ptRaw = propertyData.property_type ?? propertyData['property-type'] ?? [];
    const transaccion = Array.isArray(txRaw)
      ? txRaw.map(v => (typeof v === 'string' ? v : (v && v.nombre) ? v.nombre : '')).filter(Boolean)
      : [];
    const property_type = Array.isArray(ptRaw)
      ? ptRaw.map(v => (typeof v === 'string' ? v : (v && v.nombre) ? v.nombre : '')).filter(Boolean)
      : [];

    console.log('📊 Datos extraídos:', {
      id, title, author_email, local_user_found, can_import
    });

    // Status badges
    let statusBadges = '';
    if (already_imported) {
      statusBadges += '<span class="hpbdweb-badge hpbdweb-badge--error">❌ Ya Importada</span>';
    }
    if (!local_user_found) {
      statusBadges += '<span class="hpbdweb-badge hpbdweb-badge--error">❌ Usuario no encontrado</span>';
    }
    if (can_import) {
      statusBadges += '<span class="hpbdweb-badge hpbdweb-badge--success">✅ Lista para importar</span>';
    }

    container.innerHTML = `
      <div class="hpbdweb-editable-form">
        <div class="hpbdweb-editable-form__header">
          <h3>📋 Vista Previa Editable - ID: ${id}</h3>
          <div class="hpbdweb-editable-form__status">
            ${statusBadges}
          </div>
        </div>

        <form id="hpbdweb-editable-form" class="hpbdweb-editable-form__form">
          <input type="hidden" name="id" value="${id}">
          
          <!-- Información Básica -->
          <fieldset class="hpbdweb-fieldset">
            <legend>🏠 Información Básica</legend>
            
            <div class="hpbdweb-field-group">
              <label class="hpbdweb-label">
                <span>Título de la propiedad</span>
                <input type="text" name="title" value="${escapeHtml(title)}" class="hpbdweb-input hpbdweb-input--full" required>
              </label>
            </div>

            <div class="hpbdweb-field-group">
              <label class="hpbdweb-label">
                <span>Descripción</span>
                <textarea name="content" class="hpbdweb-textarea" rows="4">${escapeHtml(content)}</textarea>
              </label>
            </div>

            <div class="hpbdweb-field-group">
              <label class="hpbdweb-label">
                <span>Resumen (opcional)</span>
                <textarea name="excerpt" class="hpbdweb-textarea" rows="2">${escapeHtml(excerpt)}</textarea>
              </label>
            </div>

            <div class="hpbdweb-field-group">
              <label class="hpbdweb-label">
                <span>Estado de la publicación</span>
                <select name="status" class="hpbdweb-select">
                  <option value="publish" ${propertyData.status === 'publish' ? 'selected' : ''}>Publicado</option>
                  <option value="draft" ${propertyData.status === 'draft' ? 'selected' : ''}>Borrador</option>
                  <option value="private" ${propertyData.status === 'private' ? 'selected' : ''}>Privado</option>
                </select>
              </label>
            </div>
          </fieldset>

          <!-- Clasificación (Transacción y Tipo de Propiedad) -->
          <fieldset class="hpbdweb-fieldset">
            <legend>🏷️ Clasificación</legend>
            <div class="hpbdweb-field-group">
              <label class="hpbdweb-label">
                <span>Tipo de transacción *</span>
                <input type="text" name="transaccion" value="${Array.isArray(transaccion) && transaccion.length ? transaccion.join(', ') : ''}" class="hpbdweb-input hpbdweb-input--full" placeholder="Ej. Venta, Renta" required>
              </label>
            </div>
            <div class="hpbdweb-field-group">
              <label class="hpbdweb-label">
                <span>Tipo de propiedad *</span>
                <input type="text" name="property_type" value="${Array.isArray(property_type) && property_type.length ? property_type.join(', ') : ''}" class="hpbdweb-input hpbdweb-input--full" placeholder="Ej. Casa, Bodega" required>
              </label>
            </div>
          </fieldset>

          <!-- Imágenes -->
          <fieldset class="hpbdweb-fieldset">
            <legend>📷 Imágenes (IDs de BDweb)</legend>
            <div class="hpbdweb-field-group">
              <label class="hpbdweb-label">
                <span>ID de Imagen Destacada</span>
                <input type="number" name="featured_media" value="${propertyData.featured_media || ''}" class="hpbdweb-input" placeholder="Ej. 12345">
              </label>
              <small class="hpbdweb-help">Usa el ID de la imagen en BDweb (no el ID local). Se descargará y asignará como destacada.</small>
            </div>

            <div class="hpbdweb-field-group">
              <label class="hpbdweb-label">
                <span>Galería (IDs separados por coma)</span>
                <textarea name="gallery" class="hpbdweb-textarea" rows="2" placeholder="Ej. 111, 222, 333">${Array.isArray(propertyData.gallery) && propertyData.gallery.length ? propertyData.gallery.join(', ') : ''}</textarea>
              </label>
              <small class="hpbdweb-help">IDs de imágenes en BDweb separados por comas. Importaremos hasta 3 para evitar timeouts.</small>
            </div>
          </fieldset>

          <!-- Multimedia -->
          <fieldset class="hpbdweb-fieldset">
            <legend>🎬 Multimedia</legend>
            <div class="hpbdweb-field-group">
              <label class="hpbdweb-label">
                <span>Video (URL)</span>
                <input type="url" name="video" value="${escapeHtml(video || '')}" class="hpbdweb-input hpbdweb-input--full" placeholder="https://...">
              </label>
            </div>
          </fieldset>

          <!-- Información del Autor -->
          <fieldset class="hpbdweb-fieldset">
            <legend>👤 Información del Autor ${local_user_found ? '✅' : '❌'}</legend>
            
            <div class="hpbdweb-field-row">
              <label class="hpbdweb-label">
                <span>Nombre</span>
                <input type="text" name="author_name" value="${escapeHtml(author_name)}" class="hpbdweb-input">
              </label>
              <label class="hpbdweb-label">
                <span>Email ${local_user_found ? '✅' : '❌'}</span>
                <input type="email" name="author_email" value="${escapeHtml(author_email)}" class="hpbdweb-input" readonly>
              </label>
              <label class="hpbdweb-label">
                <span>Teléfono</span>
                <input type="text" name="author_phone" value="${escapeHtml(author_phone)}" class="hpbdweb-input">
              </label>
            </div>
          </fieldset>

          <!-- Características de la Propiedad -->
          <fieldset class="hpbdweb-fieldset">
            <legend>🏡 Características</legend>
            
            <div class="hpbdweb-field-row">
              <label class="hpbdweb-label">
                <span>Precio</span>
                <input type="number" name="price" value="${price}" class="hpbdweb-input" step="0.01">
              </label>
              <label class="hpbdweb-label">
                <span>Comisión</span>
                <input type="text" name="comision" value="${comision || ''}" class="hpbdweb-input" placeholder="Ej. 3% o monto">
              </label>
              <label class="hpbdweb-label">
                <span>Habitaciones</span>
                <input type="number" name="bedrooms" value="${bedrooms}" class="hpbdweb-input">
              </label>
              <label class="hpbdweb-label">
                <span>Baños</span>
                <input type="number" name="bathrooms" value="${bathrooms}" class="hpbdweb-input" step="0.5">
              </label>
            </div>

            <div class="hpbdweb-field-row">
              <label class="hpbdweb-label">
                <span>Tamaño construcción (m²)</span>
                <input type="number" name="property_size" value="${property_size}" class="hpbdweb-input" step="0.01">
              </label>
              <label class="hpbdweb-label">
                <span>Tamaño terreno (m²)</span>
                <input type="number" name="area_size" value="${area_size}" class="hpbdweb-input" step="0.01">
              </label>
              <label class="hpbdweb-label">
                <span>Año construcción</span>
                <input type="number" name="year_built" value="${year_built}" class="hpbdweb-input" min="1900" max="2030">
              </label>
            </div>

            <div class="hpbdweb-field-group">
              <label class="hpbdweb-label">
                <span>Condiciones para comisión</span>
                <textarea name="condiciones_para_comision" class="hpbdweb-textarea" rows="2">${escapeHtml(condiciones_para_comision || '')}</textarea>
              </label>
            </div>
          </fieldset>

          <!-- Medidas adicionales -->
          <fieldset class="hpbdweb-fieldset">
            <legend>📐 Medidas adicionales</legend>
            <div class="hpbdweb-field-row">
              <label class="hpbdweb-label">
                <span>Área techada (m²)</span>
                <input type="number" name="techado" value="${techado}" class="hpbdweb-input" step="0.01">
              </label>
              <label class="hpbdweb-label">
                <span>Patio de maniobras (m²)</span>
                <input type="number" name="patio_maniobras" value="${patio_maniobras}" class="hpbdweb-input" step="0.01">
              </label>
            </div>
            <div class="hpbdweb-field-row">
              <label class="hpbdweb-label">
                <span>Frente (m)</span>
                <input type="number" name="frente" value="${frente}" class="hpbdweb-input" step="0.01">
              </label>
              <label class="hpbdweb-label">
                <span>Fondo (m)</span>
                <input type="number" name="fondo" value="${fondo}" class="hpbdweb-input" step="0.01">
              </label>
              <label class="hpbdweb-label">
                <span>Uso de suelo</span>
                <input type="text" name="uso_suelo" value="${escapeHtml(uso_suelo)}" class="hpbdweb-input">
              </label>
            </div>
          </fieldset>

          <!-- Ubicación -->
          <fieldset class="hpbdweb-fieldset">
            <legend>📍 Ubicación</legend>
            
            <div class="hpbdweb-field-group">
              <label class="hpbdweb-label">
                <span>Dirección</span>
                <input type="text" name="address" value="${escapeHtml(address)}" class="hpbdweb-input hpbdweb-input--full">
              </label>
            </div>

            <div class="hpbdweb-field-group">
              <button type="button" id="hpbdweb-parse-address-btn" class="button">Obtener colonia/ciudad/estado/CP</button>
            </div>

            <div class="hpbdweb-field-row">
              <label class="hpbdweb-label">
                <span>Colonia</span>
                <input type="text" name="colonia" value="${escapeHtml(colonia)}" class="hpbdweb-input">
              </label>
              <label class="hpbdweb-label">
                <span>Ciudad</span>
                <input type="text" name="ciudad" value="${escapeHtml(ciudad)}" class="hpbdweb-input">
              </label>
              <label class="hpbdweb-label">
                <span>Estado</span>
                <input type="text" name="estado" value="${escapeHtml(estado)}" class="hpbdweb-input">
              </label>
              <label class="hpbdweb-label">
                <span>Código Postal</span>
                <input type="text" name="codigo_postal" value="${escapeHtml(codigo_postal)}" class="hpbdweb-input">
              </label>
            </div>

            <div class="hpbdweb-field-row">
              <label class="hpbdweb-label">
                <span>Latitud ${latitude ? '✅' : '🗺️'}</span>
                <input type="number" name="latitude" value="${latitude}" class="hpbdweb-input" step="any" placeholder="Se obtendrá automáticamente">
              </label>
              <label class="hpbdweb-label">
                <span>Longitud ${longitude ? '✅' : '🗺️'}</span>
                <input type="number" name="longitude" value="${longitude}" class="hpbdweb-input" step="any" placeholder="Se obtendrá automáticamente">
              </label>
            </div>

            ${(!latitude || !longitude) && address ? 
              `<div class="hpbdweb-notice hpbdweb-notice--info">🗺️ Las coordenadas GPS se obtendrán automáticamente de la dirección si no están disponibles<br>
                <button type="button" id="hpbdweb-geocode-btn" class="button">Obtener coordenadas ahora</button>
              </div>` 
              : ''}
          </fieldset>

          <!-- Amenidades -->
          <fieldset class="hpbdweb-fieldset">
            <legend>🧩 Amenidades</legend>
            <div class="hpbdweb-field-group">
              <label class="hpbdweb-label">
                <span>Listado (separado por comas)</span>
                <textarea name="amenidades" class="hpbdweb-textarea" rows="2" placeholder="Ej. Jardín, Alberca, Vigilancia">${Array.isArray(amenidades) && amenidades.length ? amenidades.join(', ') : ''}</textarea>
              </label>
              <small class="hpbdweb-help">Se crearán términos en la taxonomía "amenidades" si no existen.</small>
            </div>
          </fieldset>

          <!-- Actions -->
          <div class="hpbdweb-editable-form__actions">
            ${local_user_found ? `
              <button type="button" id="hpbdweb-import-edited-btn" class="button button-primary button-large">
                ${already_imported ? '⚠️ Importar de nuevo (duplicar)' : '🚀 Importar con estos datos'}
              </button>
            ` : `
              <button type="button" class="button button-primary button-large" disabled>
                ❌ Usuario no encontrado
              </button>
            `}
            
            <button type="button" onclick="document.getElementById('hpbdweb-property-preview').style.display='none'" class="button">
              Cancelar
            </button>
          </div>
        </form>
      </div>
    `;

    // Parse address button (colonia/ciudad/estado/CP)
    const parseAddressBtn = container.querySelector('#hpbdweb-parse-address-btn');
    if (parseAddressBtn) {
      parseAddressBtn.addEventListener('click', async () => {
        try {
          parseAddressBtn.disabled = true;
          parseAddressBtn.textContent = 'Obteniendo...';
          const form = container.querySelector('#hpbdweb-editable-form');
          const formData = new FormData(form);
          const address = String(formData.get('address') || '').trim();
          const colonia = String(formData.get('colonia') || '').trim();
          const ciudad  = String(formData.get('ciudad') || '').trim();
          const estado  = String(formData.get('estado') || '').trim();

          const full_address = [address, colonia, ciudad, estado].filter(Boolean).join(', ');
          if (!full_address) {
            logger('❌ Proporciona al menos la dirección para analizar', 'error');
            return;
          }

          const { ajax_url, nonce } = window.HPBDWEB_SETTINGS || {};
          const req = new FormData();
          req.append('action', 'hp_bdweb_parse_address');
          req.append('_wpnonce', nonce);
          req.append('full_address', full_address);

          const res = await fetch(ajax_url, { method: 'POST', credentials: 'same-origin', body: req });
          const txt = await res.text();
          let data;
          try {
            const brace = txt.indexOf('{');
            data = JSON.parse(brace > 0 ? txt.slice(brace) : txt);
          } catch (e) {
            throw new Error('Respuesta no válida del servidor: ' + txt.substring(0, 200));
          }
          if (!data.success) throw new Error(data?.data?.message || 'Error analizando dirección');

          // Fill inputs
          const setVal = (selector, val) => {
            const el = form.querySelector(selector);
            if (el && (val || val === 0)) el.value = val;
          };
          setVal('input[name="colonia"]', data.data.colonia);
          setVal('input[name="ciudad"]', data.data.ciudad);
          setVal('input[name="estado"]', data.data.estado);
          setVal('input[name="codigo_postal"]', data.data.codigo_postal);
          // Lat/Lng si vienen mejorados
          if (data.data.lat && data.data.lng) {
            setVal('input[name="latitude"]', data.data.lat);
            setVal('input[name="longitude"]', data.data.lng);
          }
          logger('✅ Dirección analizada correctamente', 'success');
        } catch (err) {
          logger('❌ Error analizando dirección: ' + (err?.message || 'Error desconocido'), 'error');
        } finally {
          parseAddressBtn.disabled = false;
          parseAddressBtn.textContent = 'Obtener colonia/ciudad/estado/CP';
        }
      });
    }

    // Geocode now button
    const geocodeBtn = container.querySelector('#hpbdweb-geocode-btn');
    if (geocodeBtn) {
      geocodeBtn.addEventListener('click', async () => {
        try {
          geocodeBtn.disabled = true;
          geocodeBtn.textContent = 'Obteniendo...';
          const form = container.querySelector('#hpbdweb-editable-form');
          const formData = new FormData(form);
          const address = String(formData.get('address') || '').trim();
          const colonia = String(formData.get('colonia') || '').trim();
          const ciudad  = String(formData.get('ciudad') || '').trim();
          const estado  = String(formData.get('estado') || '').trim();

          const full_address = [address, colonia, ciudad, estado].filter(Boolean).join(', ');
          if (!full_address) {
            logger('❌ Proporciona al menos la dirección para geocodificar', 'error');
            return;
          }

          const { ajax_url, nonce } = window.HPBDWEB_SETTINGS || {};
          const req = new FormData();
          req.append('action', 'hp_bdweb_geocode_address');
          req.append('_wpnonce', nonce);
          req.append('full_address', full_address);

          const res = await fetch(ajax_url, { method: 'POST', credentials: 'same-origin', body: req });
          const txt = await res.text();
          let data;
          try {
            const brace = txt.indexOf('{');
            data = JSON.parse(brace > 0 ? txt.slice(brace) : txt);
          } catch (e) {
            throw new Error('Respuesta no válida del servidor: ' + txt.substring(0, 200));
          }
          if (!data.success) throw new Error(data?.data?.message || 'Error geocodificando');

          // Fill inputs
          const latInput = form.querySelector('input[name="latitude"]');
          const lngInput = form.querySelector('input[name="longitude"]');
          if (latInput) latInput.value = data.data.lat;
          if (lngInput) lngInput.value = data.data.lng;
          logger('✅ Coordenadas obtenidas: ' + data.data.lat + ', ' + data.data.lng, 'success');
        } catch (err) {
          logger('❌ Error obteniendo coordenadas: ' + (err?.message || 'Error desconocido'), 'error');
        } finally {
          geocodeBtn.disabled = false;
          geocodeBtn.textContent = 'Obtener coordenadas ahora';
        }
      });
    }

    // Attach event listener for import button
    const importBtn = container.querySelector('#hpbdweb-import-edited-btn');
    if (importBtn) {
      importBtn.addEventListener('click', async () => {
        // Confirmación: si ya está importada, preguntar si desea duplicar
        let proceed = true;
        let forceDuplicate = false;
        if (already_imported) {
          proceed = confirm('Esta propiedad ya existe (título y autor coinciden). ¿Deseas importarla de nuevo y crear un duplicado?');
          if (proceed) forceDuplicate = true;
        } else {
          proceed = confirm('¿Importar esta propiedad con los datos editados?');
        }
        if (!proceed) return;

        importBtn.disabled = true;
        importBtn.innerHTML = '🚀 Importando...';

        try {
          // Collect form data
          const form = container.querySelector('#hpbdweb-editable-form');
          const formData = new FormData(form);
          const editedData = {};
          
          for (let [key, value] of formData.entries()) {
            editedData[key] = value;
          }

          // Bandera para permitir duplicados si el usuario lo autorizó
          if (forceDuplicate) {
            editedData.allow_duplicate = true;
          }

          // Normalizar campos de imágenes
          if (typeof editedData.featured_media !== 'undefined') {
            const fm = String(editedData.featured_media).trim();
            editedData.featured_media = fm ? parseInt(fm, 10) : '';
          }
          if (typeof editedData.gallery !== 'undefined') {
            const rawGallery = String(editedData.gallery || '').trim();
            if (rawGallery) {
              editedData.gallery = rawGallery
                .split(',')
                .map(s => s.trim())
                .filter(s => s && /^\d+$/.test(s))
                .map(n => parseInt(n, 10));
            } else {
              editedData.gallery = [];
            }
          }

          // Normalizar amenidades a array de strings
          if (typeof editedData.amenidades !== 'undefined') {
            const rawA = String(editedData.amenidades || '').trim();
            editedData.amenidades = rawA
              ? rawA.split(',').map(s => s.trim()).filter(Boolean)
              : [];
          }

          // Normalizar taxonomías de transacción y tipo de propiedad
          if (typeof editedData.transaccion !== 'undefined') {
            const rawT = String(editedData.transaccion || '').trim();
            editedData.transaccion = rawT ? rawT.split(',').map(s => s.trim()).filter(Boolean) : [];
          }
          if (typeof editedData.property_type !== 'undefined') {
            const rawPT = String(editedData.property_type || '').trim();
            editedData.property_type = rawPT ? rawPT.split(',').map(s => s.trim()).filter(Boolean) : [];
          }

          logger('🚀 Iniciando importación con datos editados...', 'info');
          let resp;
          try {
            resp = await importEditedProperty(editedData);
          } catch (e) {
            // Si el servidor respondió "ya existe" y el usuario no había aceptado aún, ofrecer reintento forzando duplicado
            const msg = e?.message || '';
            if (/ya fue importada/i.test(msg) && !forceDuplicate) {
              if (confirm('Ya existe una propiedad con este título y autor. ¿Forzar importación y crear duplicado?')) {
                editedData.allow_duplicate = true;
                resp = await importEditedProperty(editedData);
              } else {
                throw e;
              }
            } else {
              throw e;
            }
          }
          
          logger('✅ ' + resp.data.message, 'success');
          if (resp.data.edit_link) {
            logger('📝 Editar: ' + resp.data.edit_link, 'info');
          }
          if (resp.data.view_link) {
            logger('👀 Ver: ' + resp.data.view_link, 'info');
          }

          // Hide form after successful import
          container.style.display = 'none';
          
        } catch (err) {
          const message = err?.data?.message || err?.message || 'Error desconocido';
          logger('❌ Error en importación: ' + message, 'error');
          // Fallback: si el error fue por HTML colado antes del JSON, intentamos verificar si el post se creó
          try {
            if (message.startsWith('JSON Parse Error') || /Respuesta no válida/.test(message)) {
              const { checkImport } = await import('./modules/ajax.js');
              const title = String(editedData.title || '').trim();
              const authorEmail = String(editedData.author_email || '').trim();
              if (title && authorEmail) {
                const chk = await checkImport(title, authorEmail);
                if (chk?.success && chk.data?.found) {
                  logger('✅ Importación verificada aunque la respuesta JSON venía con HTML.', 'success');
                  if (chk.data.edit_link) logger('📝 Editar: ' + chk.data.edit_link, 'info');
                  if (chk.data.view_link) logger('👀 Ver: ' + chk.data.view_link, 'info');
                  const t = chk.data.taxonomies || {};
                  logger('🏷️ Taxonomías asignadas: ' + JSON.stringify(t), 'info');
                  const m = chk.data.meta || {};
                  logger('🧩 Metas guardadas: ' + JSON.stringify(m), 'info');
                  container.style.display = 'none';
                }
              }
            }
          } catch (e2) {
            logger('ℹ️ No se pudo verificar importación automáticamente: ' + (e2?.message || e2), 'info');
          }
        } finally {
          importBtn.disabled = false;
          importBtn.innerHTML = '🚀 Importar con estos datos';
        }
      });
    }
  }

  // Helper function to escape HTML
  function escapeHtml(text) {
    if (!text && text !== 0) return '';
    return String(text)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#039;');
  }

  log('Sistema de importación BDweb con vista previa editable iniciado ✅');
});
