// Base AJAX function
async function makeAjaxRequest(action, extraData = {}) {
  const { ajax_url, nonce } = window.HPBDWEB_SETTINGS || {};
  if (!ajax_url || !nonce) throw new Error('Config no disponible.');

  const form = new FormData();
  form.append('action', action);
  form.append('_wpnonce', nonce);
  
  // Add extra data
  Object.keys(extraData).forEach(key => {
    form.append(key, extraData[key]);
  });

  const res = await fetch(ajax_url, {
    method: 'POST',
    credentials: 'same-origin',
    body: form
  });

  const text = await res.text();
  
  // Debug: Log respuesta cruda
  console.log('AJAX Response Debug:', {
    action,
    status: res.status,
    statusText: res.statusText,
    headers: Object.fromEntries(res.headers.entries()),
    textLength: text.length,
    textPreview: text.substring(0, 500)
  });
  
  let data;
  try {
    // Si viene HTML antes del JSON (errores de otros plugins), intentar sanear
    let sanitized = text;
    const firstBrace = sanitized.indexOf('{');
    if (firstBrace > 0) {
      sanitized = sanitized.slice(firstBrace);
    }
    data = JSON.parse(sanitized);
  } catch (e) {
    console.error('JSON Parse Error:', {
      action,
      parseError: e.message,
      responseText: text,
      responseLength: text.length
    });
    throw new Error(`JSON Parse Error: ${e.message}. Response: ${text.substring(0, 200)}...`);
  }

  if (!res.ok || !data || !data.success) {
    // Extraer información detallada del error
    let errorMsg = 'Error desconocido';
    let errorDetails = '';
    
    if (data && data.data) {
      if (data.data.message) {
        errorMsg = data.data.message;
      }
      if (data.data.error_code) {
        errorDetails += ` (Código: ${data.data.error_code})`;
      }
      if (data.data.error_details) {
        errorDetails += ` Detalles: ${JSON.stringify(data.data.error_details)}`;
      }
    } else if (data && data.message) {
      errorMsg = data.message;
    }
    
    // Log completo para debugging
    console.error('AJAX Error Details:', {
      action,
      response: res,
      data: data,
      text: text
    });
    
    throw new Error(errorMsg + errorDetails);
  }
  
  return data;
}

// Original test function
export async function testAjax() {
  return makeAjaxRequest('hp_bdweb_ajax_test');
}

// Test BDweb API connection
export async function testBdwebApi() {
  return makeAjaxRequest('hp_bdweb_test_api');
}

// Load properties list
export async function loadProperties() {
  return makeAjaxRequest('hp_bdweb_load_properties');
}

// Preview a specific property
export async function previewProperty(propertyId) {
  return makeAjaxRequest('hp_bdweb_preview_property', {
    property_id: propertyId
  });
}

// Get editable property data
export async function getEditableProperty(propertyId) {
  return makeAjaxRequest('hp_bdweb_get_editable_property', {
    property_id: propertyId
  });
}

// Import property with edited data
export async function importEditedProperty(editedData) {
  // Enviar los datos editados como JSON para evitar '[object Object]' en PHP
  return makeAjaxRequest('hp_bdweb_import_edited_property', {
    edited_data: JSON.stringify(editedData)
  });
}

// Import a property
export async function importProperty(propertyId) {
  return makeAjaxRequest('hp_bdweb_import_property', {
    property_id: propertyId
  });
}

// Import all properties (by page)
export async function importAllBatch(page = 1) {
  return makeAjaxRequest('hp_bdweb_import_all', {
    page: page
  });
}

// Load properties with pagination
export async function loadPropertiesWithPage(page = 1) {
  return makeAjaxRequest('hp_bdweb_load_properties', {
    page: page
  });
}

// Import a specific page
export async function importPage(page = 1) {
  return makeAjaxRequest('hp_bdweb_import_page', {
    page: page
  });
}

// Batch import (small batches with offset)
export async function batchImport(offset = 0) {
  return makeAjaxRequest('hp_bdweb_batch_import', {
    offset: offset
  });
}

// Debug property data structure
export async function debugProperty(propertyId) {
  return makeAjaxRequest('hp_bdweb_debug_property', {
    property_id: propertyId
  });
}

// Check if import succeeded by querying server for post by title and author email
export async function checkImport(title, authorEmail) {
  return makeAjaxRequest('hp_bdweb_check_import', {
    title: title,
    author_email: authorEmail
  });
}

// Verificación de exportaciones (para debug)
console.log('📦 Módulo ajax.js cargado con exportaciones:', {
  testAjax: typeof testAjax,
  testBdwebApi: typeof testBdwebApi,
  loadProperties: typeof loadProperties,
  previewProperty: typeof previewProperty,
  getEditableProperty: typeof getEditableProperty,
  importEditedProperty: typeof importEditedProperty,
  importProperty: typeof importProperty,
  loadPropertiesWithPage: typeof loadPropertiesWithPage,
  importPage: typeof importPage,
  batchImport: typeof batchImport,
  debugProperty: typeof debugProperty,
  checkImport: typeof checkImport
});