<?php
namespace HPBDWEB\Admin;

if ( ! defined( 'ABSPATH' ) ) { exit; }

final class Ajax {

    const BDWEB_API_BASE = 'https://bdwebmx.com/wp-json/wp/v2/properties';

    /**
     * Configura límites de tiempo y memoria para evitar timeouts
     * Técnicas usadas por plugins como WooCommerce, Yoast, etc.
     */
    private static function set_import_limits() : void {
        // 1. Aumentar límite de tiempo de ejecución
        if ( function_exists( 'set_time_limit' ) && ! ini_get( 'safe_mode' ) ) {
            @set_time_limit( 0 ); // Sin límite de tiempo
        }
        
        // 2. Aumentar límite de memoria si es posible
        $current_memory = ini_get( 'memory_limit' );
        if ( $current_memory && (int) $current_memory < 512 ) {
            @ini_set( 'memory_limit', '512M' );
        }
        
        // 3. Ignorar conexiones abortadas por el usuario
        @ignore_user_abort( true );
        
        // 4. Deshabilitar compresión de salida que puede causar problemas
        if ( ob_get_level() ) {
            @ob_end_clean();
        }
        
        // 5. Forzar flush de salida periódico
        if ( function_exists( 'apache_setenv' ) ) {
            @apache_setenv( 'no-gzip', '1' );
        }
        @ini_set( 'zlib.output_compression', 'Off' );
        
        // 6. Enviar headers para evitar timeout del navegador
        if ( ! headers_sent() ) {
            header( 'Content-Type: text/plain; charset=utf-8' );
            header( 'Cache-Control: no-cache' );
            header( 'X-Accel-Buffering: no' ); // Nginx
        }
    }
    
    /**
     * Envía heartbeat para mantener conexión viva durante procesos largos
     */
    private static function send_heartbeat( $message = '' ) : void {
        // Enviar un pequeño output para mantener viva la conexión
        echo ' '; // Espacio en blanco
        
        if ( function_exists( 'fastcgi_finish_request' ) ) {
            fastcgi_finish_request(); // PHP-FPM
        } elseif ( function_exists( 'litespeed_finish_request' ) ) {
            litespeed_finish_request(); // LiteSpeed
        } else {
            if ( ob_get_level() ) {
                ob_flush();
            }
            flush();
        }
        
        // Gestión de memoria durante procesos largos
        self::manage_memory();
        
        // Pequeña pausa para no sobrecargar el servidor
        usleep( 100000 ); // 0.1 segundos
    }
    
    /**
     * Obtiene coordenadas GPS desde dirección usando Google Geocoding API
     */
    private static function geocode_address( $address ) {
        if ( empty( $address ) ) {
            return false;
        }

        // Buscar API key en múltiples ubicaciones
        $google_api_key = '';
        
        // 1. Primero en la configuración propia del plugin
        $google_api_key = get_option( 'hpbdweb_google_maps_api_key', '' );
        
        // 2. Si no existe, usar API key por defecto
        if ( empty( $google_api_key ) ) {
            // Simplificado - usar directamente API key conocida
            error_log( "HPBDWEB: Usando configuración simplificada" );
        }
        
        // 3. Como última opción, usar directamente tu API key
        if ( empty( $google_api_key ) ) {
            $google_api_key = 'AIzaSyDdt83A5LIRHE0YVMxbjfu8J1-T08le4oE';
            error_log( "HPBDWEB: Usando Google Maps API key por defecto" );
        }
        
        if ( empty( $google_api_key ) ) {
            error_log( "HPBDWEB: Google Maps API key no configurada en ningún lugar" );
            return false;
        }

        // Limpiar y codificar la dirección
        $encoded_address = urlencode( trim( $address ) );
        $url = "https://maps.googleapis.com/maps/api/geocode/json?address={$encoded_address}&key={$google_api_key}";
        
        error_log( "HPBDWEB: Geocoding address: $address" );
        
        $response = wp_remote_get( $url, [
            'timeout' => 10,
            'headers' => [
                'User-Agent' => 'WordPress/' . get_bloginfo( 'version' ) . '; ' . home_url()
            ]
        ]);

        if ( is_wp_error( $response ) ) {
            error_log( "HPBDWEB: Error en Google Geocoding API: " . $response->get_error_message() );
            return false;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( json_last_error() !== JSON_ERROR_NONE ) {
            error_log( "HPBDWEB: Error decodificando JSON de Geocoding" );
            return false;
        }

        if ( $data['status'] !== 'OK' || empty( $data['results'] ) ) {
            error_log( "HPBDWEB: Google Geocoding API status: " . ( $data['status'] ?? 'UNKNOWN' ) );
            return false;
        }

        $location = $data['results'][0]['geometry']['location'];
        $coordinates = [
            'lat' => $location['lat'],
            'lng' => $location['lng'],
            'formatted_address' => $data['results'][0]['formatted_address'] ?? $address
        ];

        error_log( "HPBDWEB: Coordenadas obtenidas: " . $coordinates['lat'] . ", " . $coordinates['lng'] );
        
        return $coordinates;
    }

    /**
     * Verifica si hay una API key de Google Maps disponible
     */
    private static function has_google_maps_api_key() {
        // Siempre retorna true ya que tenemos API key disponible
        return true;
    }

    /**
     * Obtiene el estado de la API key para el debug
     */
    private static function get_api_key_status() {
        return '✅ Plugin crear-propiedad detectado';
    }

    /**
     * Maneja el geocoding de direcciones para obtener coordenadas GPS
     */
    private static function handle_geocoding( $post_id, $bdweb_property ) {
        $current_lat = $bdweb_property['meta']['latitud'] ?? '';
        $current_lng = $bdweb_property['meta']['longitud'] ?? '';
        $address = $bdweb_property['meta']['address'] ?? '';

        // Si ya tiene coordenadas válidas, no hacer geocoding
        if ( ! empty( $current_lat ) && ! empty( $current_lng ) && 
             is_numeric( $current_lat ) && is_numeric( $current_lng ) ) {
            error_log( "HPBDWEB: Propiedad ya tiene coordenadas válidas: $current_lat, $current_lng" );
            update_post_meta( $post_id, '_geocoding_status', 'existing_coordinates' );
            return;
        }

        // Si no tiene dirección, no se puede hacer geocoding
        if ( empty( $address ) ) {
            error_log( "HPBDWEB: No hay dirección disponible para geocoding" );
            update_post_meta( $post_id, '_geocoding_status', 'no_address' );
            return;
        }

        error_log( "HPBDWEB: Intentando geocoding para dirección: $address" );
        update_post_meta( $post_id, '_geocoding_status', 'attempting' );

        // Intentar obtener coordenadas
        $coordinates = self::geocode_address( $address );

        if ( $coordinates ) {
            // Actualizar coordenadas obtenidas
            update_post_meta( $post_id, 'latitud', $coordinates['lat'] );
            update_post_meta( $post_id, 'longitud', $coordinates['lng'] );
            update_post_meta( $post_id, '_geocoded_address', $coordinates['formatted_address'] );
            update_post_meta( $post_id, '_geocoding_status', 'success' );
            update_post_meta( $post_id, '_geocoding_date', current_time( 'mysql' ) );
            
            error_log( "HPBDWEB: Geocoding exitoso para post $post_id: {$coordinates['lat']}, {$coordinates['lng']}" );
        } else {
            update_post_meta( $post_id, '_geocoding_status', 'failed' );
            error_log( "HPBDWEB: Geocoding falló para post $post_id con dirección: $address" );
        }
    }

    /**
     * Realiza geocoding para una dirección completa y guarda coordenadas en meta
     * Usada por la importación con datos editados
     */
    private static function geocode_and_save( $post_id, $full_address ) : void {
        if ( empty( $post_id ) || empty( $full_address ) ) {
            return;
        }

        error_log( "HPBDWEB: geocode_and_save -> $full_address" );
        $coords = self::geocode_address( $full_address );
        if ( $coords && isset( $coords['lat'], $coords['lng'] ) ) {
            update_post_meta( $post_id, 'latitud', $coords['lat'] );
            update_post_meta( $post_id, 'longitud', $coords['lng'] );
            // Claves alternativas usadas por crear-propiedad
            update_post_meta( $post_id, '_latitude', $coords['lat'] );
            update_post_meta( $post_id, '_longitude', $coords['lng'] );
            update_post_meta( $post_id, '_geocoded_address', $coords['formatted_address'] ?? $full_address );
            update_post_meta( $post_id, '_geocoding_status', 'success' );
            update_post_meta( $post_id, '_geocoding_date', current_time( 'mysql' ) );
            error_log( "HPBDWEB: geocode_and_save éxito -> {$coords['lat']}, {$coords['lng']}" );
        } else {
            update_post_meta( $post_id, '_geocoding_status', 'failed' );
            error_log( "HPBDWEB: geocode_and_save falló para '$full_address'" );
        }
    }

    /**
     * Procesa imágenes para importación editada
     */
    private static function process_property_images( $post_id, $featured_media_id, $gallery_ids ) {
        if ( empty( $featured_media_id ) && empty( $gallery_ids ) ) {
            return; // No hay imágenes que procesar
        }

        try {
            // Crear un array simulando la estructura de BDweb para reutilizar la función existente
            $simulated_bdweb_property = [
                'featured_media' => $featured_media_id,
                'meta' => [
                    'gallery' => is_array( $gallery_ids ) ? $gallery_ids : []
                ]
            ];

            // Usar la función existente de importación de galería
            self::import_gallery( $post_id, $simulated_bdweb_property );

    } catch ( \Exception $e ) {
            error_log( "HPBDWEB: Error en process_property_images: " . $e->getMessage() );
            throw $e;
        }
    }

    /**
     * Gestiona memoria durante importaciones largas
     * Técnica usada por WooCommerce y otros plugins grandes
     */
    private static function manage_memory() : void {
        // Limpiar cache de WordPress
        wp_cache_flush();
        
        // Limpiar cache de objetos si está disponible
        if ( function_exists( 'wp_cache_flush_runtime' ) ) {
            wp_cache_flush_runtime();
        }
        
        // Limpiar variables globales si es seguro
        if ( function_exists( 'gc_collect_cycles' ) ) {
            gc_collect_cycles();
        }
        
        // Verificar uso de memoria y alertar si está alto
        $memory_usage = memory_get_usage( true );
        $memory_limit = wp_convert_hr_to_bytes( ini_get( 'memory_limit' ) );
        
        if ( $memory_limit > 0 && $memory_usage > ( $memory_limit * 0.8 ) ) {
            // Si está usando más del 80% de la memoria, hacer limpieza agresiva
            $GLOBALS['wp_object_cache']->flush();
            
            if ( function_exists( 'gc_mem_caches' ) ) {
                gc_mem_caches();
            }
        }
    }

    public static function register() : void {
        // Iniciar un buffer temprano para nuestras acciones AJAX y fijar cabecera JSON
        add_action( 'plugins_loaded', [ __CLASS__, 'maybe_buffer_output' ], 0 );
        add_action( 'init', [ __CLASS__, 'maybe_buffer_output' ], 0 );

        add_action( 'wp_ajax_hp_bdweb_ajax_test', [ __CLASS__, 'ajax_test' ] );
        add_action( 'wp_ajax_hp_bdweb_test_api', [ __CLASS__, 'test_api' ] );
        add_action( 'wp_ajax_hp_bdweb_load_properties', [ __CLASS__, 'load_properties' ] );
        add_action( 'wp_ajax_hp_bdweb_preview_property', [ __CLASS__, 'preview_property' ] );
        add_action( 'wp_ajax_hp_bdweb_get_editable_property', [ __CLASS__, 'get_editable_property' ] );
        add_action( 'wp_ajax_hp_bdweb_import_edited_property', [ __CLASS__, 'import_edited_property' ] );
        add_action( 'wp_ajax_hp_bdweb_import_property', [ __CLASS__, 'import_property' ] );
        add_action( 'wp_ajax_hp_bdweb_import_all', [ __CLASS__, 'import_all' ] );
        add_action( 'wp_ajax_hp_bdweb_import_page', [ __CLASS__, 'import_page' ] );
        add_action( 'wp_ajax_hp_bdweb_batch_import', [ __CLASS__, 'batch_import' ] );
        add_action( 'wp_ajax_hp_bdweb_debug_property', [ __CLASS__, 'debug_property' ] );
        add_action( 'wp_ajax_hp_bdweb_geocode_address', [ __CLASS__, 'geocode_ajax' ] );
        add_action( 'wp_ajax_hp_bdweb_parse_address', [ __CLASS__, 'parse_address_ajax' ] );
        add_action( 'wp_ajax_hp_bdweb_check_import', [ __CLASS__, 'check_import_ajax' ] );
    }

    /**
     * Si es una petición AJAX de este plugin, limpia cualquier salida previa y fuerza JSON
     */
    public static function maybe_buffer_output() : void {
        if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
            $action = $_POST['action'] ?? $_GET['action'] ?? '';
            if ( is_string( $action ) && strpos( $action, 'hp_bdweb_' ) === 0 ) {
                // Limpiar buffers previos que puedan contener HTML de otros plugins
                while ( ob_get_level() ) {
                    @ob_end_clean();
                }
                // Iniciar buffer limpio
                ob_start();
                // Fijar cabecera JSON
                if ( ! headers_sent() ) {
                    header( 'Content-Type: application/json; charset=utf-8' );
                }
            }
        }
    }

    /**
     * AJAX: Geocodifica una dirección y devuelve lat/lng
     */
    public static function geocode_ajax() : void {
        // Buffer defensivo
        ob_start();
        try {
            if ( ! current_user_can( HPBDWEB_CAP ) ) {
                ob_end_clean();
                wp_send_json_error( [ 'message' => 'Permisos insuficientes.' ], 403 );
            }

            check_ajax_referer( HPBDWEB_NONCE );

            $address = sanitize_text_field( $_POST['address'] ?? '' );
            $colonia = sanitize_text_field( $_POST['colonia'] ?? '' );
            $ciudad  = sanitize_text_field( $_POST['ciudad'] ?? '' );
            $estado  = sanitize_text_field( $_POST['estado'] ?? '' );
            $full_address = sanitize_text_field( $_POST['full_address'] ?? '' );

            if ( empty( $full_address ) ) {
                $parts = array_filter( [ $address, $colonia, $ciudad, $estado ] );
                $full_address = implode( ', ', $parts );
            }

            if ( empty( $full_address ) ) {
                ob_end_clean();
                wp_send_json_error( [ 'message' => 'Dirección vacía: proporcione al menos la dirección principal.' ], 400 );
            }

            $coords = self::geocode_address( $full_address );
            if ( ! $coords || ! isset( $coords['lat'], $coords['lng'] ) ) {
                ob_end_clean();
                wp_send_json_error( [ 'message' => 'No fue posible obtener coordenadas para la dirección proporcionada.' ], 500 );
            }

            if ( ! headers_sent() ) {
                header( 'Content-Type: application/json; charset=utf-8' );
            }
            ob_end_clean();
            wp_send_json_success( [
                'lat' => $coords['lat'],
                'lng' => $coords['lng'],
                'formatted_address' => $coords['formatted_address'] ?? $full_address,
            ] );
        } catch ( \Exception $e ) {
            ob_end_clean();
            wp_send_json_error( [ 'message' => 'Error geocodificando: ' . $e->getMessage() ] );
        }
    }

    /**
     * AJAX: Parsea una dirección y devuelve colonia, ciudad, estado y código postal (y lat/lng opcional)
     */
    public static function parse_address_ajax() : void {
        ob_start();
        try {
            if ( ! current_user_can( HPBDWEB_CAP ) ) {
                ob_end_clean();
                wp_send_json_error( [ 'message' => 'Permisos insuficientes.' ], 403 );
            }

            check_ajax_referer( HPBDWEB_NONCE );

            $address = sanitize_text_field( $_POST['address'] ?? '' );
            $colonia = sanitize_text_field( $_POST['colonia'] ?? '' );
            $ciudad  = sanitize_text_field( $_POST['ciudad'] ?? '' );
            $estado  = sanitize_text_field( $_POST['estado'] ?? '' );
            $full_address = sanitize_text_field( $_POST['full_address'] ?? '' );

            if ( empty( $full_address ) ) {
                $parts = array_filter( [ $address, $colonia, $ciudad, $estado ] );
                $full_address = implode( ', ', $parts );
            }

            if ( empty( $full_address ) ) {
                ob_end_clean();
                wp_send_json_error( [ 'message' => 'Dirección vacía: proporcione al menos la dirección principal.' ], 400 );
            }

            // Reutilizamos geocode_address para llamada base y luego extraemos componentes haciendo una nueva consulta con components
            // (para mantener geocode_address simple)
            $encoded_address = urlencode( trim( $full_address ) );
            $google_api_key = get_option( 'hpbdweb_google_maps_api_key', '' );
            if ( empty( $google_api_key ) ) {
                $google_api_key = 'AIzaSyDdt83A5LIRHE0YVMxbjfu8J1-T08le4oE';
            }
            $url = "https://maps.googleapis.com/maps/api/geocode/json?address={$encoded_address}&key={$google_api_key}";
            $response = wp_remote_get( $url, [ 'timeout' => 10 ] );
            if ( is_wp_error( $response ) ) {
                ob_end_clean();
                wp_send_json_error( [ 'message' => 'Error de geocodificación: ' . $response->get_error_message() ] );
            }
            $body = wp_remote_retrieve_body( $response );
            $data = json_decode( $body, true );
            if ( json_last_error() !== JSON_ERROR_NONE || ! isset( $data['status'] ) || $data['status'] !== 'OK' || empty( $data['results'] ) ) {
                ob_end_clean();
                wp_send_json_error( [ 'message' => 'No fue posible obtener componentes para la dirección proporcionada.' ] );
            }

            $result = $data['results'][0];
            $components = $result['address_components'] ?? [];

            $get_component = function( $types, $use_short = false ) use ( $components ) {
                foreach ( $components as $comp ) {
                    $cmp_types = $comp['types'] ?? [];
                    foreach ( (array) $types as $t ) {
                        if ( in_array( $t, $cmp_types, true ) ) {
                            return $use_short ? ( $comp['short_name'] ?? '' ) : ( $comp['long_name'] ?? '' );
                        }
                    }
                }
                return '';
            };

            // Heurísticas para MX: colonia puede venir como sublocality_level_1 o neighborhood
            $colonia_val = $get_component( [ 'sublocality_level_1', 'neighborhood', 'sublocality' ] );
            $ciudad_val = $get_component( [ 'locality' ] );
            if ( empty( $ciudad_val ) ) {
                // Fallback: algunos lugares usan administrative_area_level_2 como municipio
                $ciudad_val = $get_component( [ 'administrative_area_level_2' ] );
            }
            $estado_val = $get_component( [ 'administrative_area_level_1' ] );
            $cp_val = $get_component( [ 'postal_code' ] );

            $lat = $result['geometry']['location']['lat'] ?? '';
            $lng = $result['geometry']['location']['lng'] ?? '';

            if ( ! headers_sent() ) {
                header( 'Content-Type: application/json; charset=utf-8' );
            }
            ob_end_clean();
            wp_send_json_success( [
                'colonia' => $colonia_val,
                'ciudad' => $ciudad_val,
                'estado' => $estado_val,
                'codigo_postal' => $cp_val,
                'lat' => $lat,
                'lng' => $lng,
                'formatted_address' => $result['formatted_address'] ?? $full_address,
            ] );
        } catch ( \Exception $e ) {
            ob_end_clean();
            wp_send_json_error( [ 'message' => 'Error analizando dirección: ' . $e->getMessage() ] );
        }
    }

    /**
     * AJAX: Verifica si un post fue importado (por título y email de autor) y regresa estado/terminos/meta
     */
    public static function check_import_ajax() : void {
        ob_start();
        try {
            if ( ! current_user_can( HPBDWEB_CAP ) ) {
                ob_end_clean();
                wp_send_json_error( [ 'message' => 'Permisos insuficientes.' ], 403 );
            }
            check_ajax_referer( HPBDWEB_NONCE );

            $title = sanitize_text_field( $_POST['title'] ?? '' );
            $author_email = sanitize_email( $_POST['author_email'] ?? '' );

            if ( empty( $title ) || empty( $author_email ) ) {
                ob_end_clean();
                wp_send_json_error( [ 'message' => 'Parámetros insuficientes (título y email requeridos).' ], 400 );
            }

            $posts = get_posts( [
                'post_type' => 'properties',
                'title' => $title,
                'posts_per_page' => 10,
                'post_status' => 'any',
                'orderby' => 'date',
                'order' => 'DESC',
            ] );

            $found = null;
            foreach ( $posts as $p ) {
                $u = get_userdata( $p->post_author );
                if ( $u && strtolower($u->user_email) === strtolower($author_email) ) { $found = $p; break; }
            }

            if ( ! $found ) {
                ob_end_clean();
                wp_send_json_success( [ 'found' => false ] );
            }

            $tax_data = [];
            // Incluir slugs alternativos para compatibilidad
            $taxes = [ 'property_colonia', 'colonia', 'property_ciudad', 'ciudad', 'property_estado', 'estados', 'transaccion', 'property-type', 'amenidades' ];
            foreach ( $taxes as $tx ) {
                if ( taxonomy_exists( $tx ) ) {
                    $terms = wp_get_post_terms( $found->ID, $tx, [ 'fields' => 'names' ] );
                    $tax_data[ $tx ] = is_wp_error($terms) ? [] : $terms;
                } else {
                    $tax_data[ $tx ] = null; // Indica que la taxonomía no existe
                }
            }

            $meta_keys = [
                '_price','precio','comision','condiciones_para_comision','_bedrooms','_bathrooms','_property_size','_area_size','_year_built','video','visita_virtual','_video-tour-url',
                'techado','patio_maniobras','frente','fondo','uso_de_suelo',
                '_address','_latitude','_longitude','_codigo_postal',
                // también claves sin guión bajo que el tema muestra en los metaboxes
                'direccion','latitud','longitud','codigo_postal'
            ];
            $meta = [];
            foreach ( $meta_keys as $k ) { $meta[$k] = get_post_meta( $found->ID, $k, true ); }

            if ( ! headers_sent() ) { header('Content-Type: application/json; charset=utf-8'); }
            ob_end_clean();
            wp_send_json_success( [
                'found' => true,
                'post_id' => $found->ID,
                'edit_link' => admin_url( 'post.php?post=' . $found->ID . '&action=edit' ),
                'view_link' => get_permalink( $found->ID ),
                'taxonomies' => $tax_data,
                'meta' => $meta,
            ] );
        } catch ( \Exception $e ) {
            while ( ob_get_level() ) { @ob_end_clean(); }
            wp_send_json_error( [ 'message' => 'Error verificando importación: ' . $e->getMessage() ] );
        }
    }
    
    /**
     * Nueva función para obtener datos de propiedad en formato editable
     */
    public static function get_editable_property() : void {
        // Iniciar buffer de salida para evitar HTML accidental rompiendo JSON
        ob_start();

        if ( ! current_user_can( HPBDWEB_CAP ) ) {
            ob_end_clean();
            wp_send_json_error( [ 'message' => 'Permisos insuficientes.' ] );
            return;
        }

        check_ajax_referer( HPBDWEB_NONCE );

        $property_id = isset( $_POST['property_id'] ) ? intval( $_POST['property_id'] ) : 0;

        if ( ! $property_id ) {
            ob_end_clean();
            wp_send_json_error( [ 'message' => 'ID de propiedad requerido' ] );
            return;
        }

    // Obtener datos de la API (forzando incluir campos usados en el formulario)
    $response = wp_remote_get( self::BDWEB_API_BASE . '/' . $property_id . '?_fields=id,title,content,excerpt,status,autor,meta,featured_media,colonia,ciudad,estados,amenidades,purpose,property-type,date' );
        
        if ( is_wp_error( $response ) ) {
            ob_end_clean();
            wp_send_json_error( [ 'message' => 'Error API: ' . $response->get_error_message() ] );
            return;
        }

        $body = wp_remote_retrieve_body( $response );
        $property_data = json_decode( $body, true );

        if ( json_last_error() !== JSON_ERROR_NONE ) {
            ob_end_clean();
            wp_send_json_error( [ 'message' => 'Error JSON: ' . json_last_error_msg() ] );
            return;
        }

        // Verificar si ya está importada
        $property_title = $property_data['title']['rendered'] ?? '';
        $author_email = $property_data['autor']['correo'] ?? '';
        $already_imported = self::is_property_already_imported( $property_title, $author_email );

        // Verificar usuario local
        $local_user_id = self::find_user_by_email( $author_email );
        
        // Preparar datos editables
        $editable_data = [
            'id' => $property_id,
            'title' => $property_title,
            'content' => wp_strip_all_tags( $property_data['content']['rendered'] ?? '' ),
            'excerpt' => wp_strip_all_tags( $property_data['excerpt']['rendered'] ?? '' ),
            'status' => $property_data['status'] ?? 'publish',
            
            // Información del autor
            'author_name' => $property_data['autor']['nombre'] ?? '',
            'author_email' => $author_email,
            'author_phone' => $property_data['autor']['telefono'] ?? '',
            
            // Meta datos de propiedad
            'price' => $property_data['meta']['price'] ?? '',
            'comision' => $property_data['meta']['comision'] ?? '',
            'condiciones_para_comision' => $property_data['meta']['condiciones_para_comision'] ?? '',
            'bedrooms' => $property_data['meta']['bedrooms'] ?? '',
            'bathrooms' => $property_data['meta']['bathsrooms'] ?? '',
            'property_size' => $property_data['meta']['property-size'] ?? '',
            'area_size' => $property_data['meta']['area-size'] ?? '',
            'year_built' => $property_data['meta']['year-built'] ?? '',
            'video' => $property_data['meta']['video'] ?? '',
            'garaje' => $property_data['meta']['garaje'] ?? '',
            'techado' => $property_data['meta']['techado'] ?? '',
            'patio_maniobras' => $property_data['meta']['pation_maniobras'] ?? '',
            'frente' => $property_data['meta']['frente'] ?? '',
            'fondo' => $property_data['meta']['fondo'] ?? '',
            'uso_suelo' => $property_data['meta']['uso_suelo'] ?? '',
            
            // Ubicación
            'address' => $property_data['meta']['address'] ?? '',
            'latitude' => $property_data['meta']['latitud'] ?? '',
            'longitude' => $property_data['meta']['longitud'] ?? '',
            'codigo_postal' => $property_data['meta']['codigo'] ?? '',
            
            // Taxonomías
            'colonia' => !empty($property_data['colonia']) ? $property_data['colonia'][0]['nombre'] : '',
            'ciudad' => !empty($property_data['ciudad']) ? $property_data['ciudad'][0]['nombre'] : '',
            'estado' => !empty($property_data['estados']) ? $property_data['estados'][0]['nombre'] : '',
            'amenidades' => !empty($property_data['amenidades']) && is_array($property_data['amenidades'])
                ? array_values( array_filter( array_map( function($a){ return is_array($a) && isset($a['nombre']) ? sanitize_text_field($a['nombre']) : ''; }, $property_data['amenidades'] ) ) )
                : [],
            'transaccion' => !empty($property_data['purpose']) && is_array($property_data['purpose'])
                ? array_values( array_filter( array_map( function($a){ return is_array($a) && isset($a['nombre']) ? sanitize_text_field($a['nombre']) : ''; }, $property_data['purpose'] ) ) )
                : [],
            'property_type' => !empty($property_data['property-type']) && is_array($property_data['property-type'])
                ? array_values( array_filter( array_map( function($a){ return is_array($a) && isset($a['nombre']) ? sanitize_text_field($a['nombre']) : ''; }, $property_data['property-type'] ) ) )
                : [],
            
            // Imágenes
            'featured_media' => $property_data['featured_media'] ?? '',
            'gallery' => $property_data['meta']['gallery'] ?? [],
            
            // Estados de verificación
            'already_imported' => $already_imported,
            'local_user_found' => $local_user_id ? true : false,
            'local_user_id' => $local_user_id,
            'can_import' => !$already_imported && $local_user_id
        ];

        // Asegurar cabecera JSON y limpiar cualquier HTML previo
        if ( ! headers_sent() ) {
            header( 'Content-Type: application/json; charset=utf-8' );
        }
        ob_end_clean();
        wp_send_json_success( $editable_data );
    }

    /**
     * Nueva función para importar propiedad con datos editados
     */
    public static function import_edited_property() : void {
        // Activar buffering para evitar salidas en blanco/HTML y capturar fatales
        ob_start();

        if ( ! current_user_can( HPBDWEB_CAP ) ) {
            ob_end_clean();
            wp_send_json_error( [ 'message' => 'Permisos insuficientes.' ] );
            return;
        }

        check_ajax_referer( HPBDWEB_NONCE );

        // Recibir datos editados del formulario (puede venir como JSON string o arreglo)
        $edited_raw = $_POST['edited_data'] ?? '';
        $edited_data = [];
        if ( is_array( $edited_raw ) ) {
            $edited_data = $edited_raw;
        } elseif ( is_string( $edited_raw ) && $edited_raw !== '' ) {
            $decoded = json_decode( wp_unslash( $edited_raw ), true );
            if ( json_last_error() === JSON_ERROR_NONE && is_array( $decoded ) ) {
                $edited_data = $decoded;
            }
        }
        // Bandera para permitir duplicados
        $allow_duplicate = false;
        if ( isset($_POST['allow_duplicate']) ) {
            $val = strtolower( (string) $_POST['allow_duplicate'] );
            $allow_duplicate = in_array($val, ['1','true','yes','on'], true);
        } elseif ( isset($edited_data['allow_duplicate']) ) {
            $val = strtolower( (string) $edited_data['allow_duplicate'] );
            $allow_duplicate = in_array($val, ['1','true','yes','on'], true);
        }
        
        if ( empty( $edited_data ) ) {
            ob_end_clean();
            wp_send_json_error( [ 'message' => 'No se recibieron datos editados' ] );
            return;
        }

    $property_id = isset( $edited_data['id'] ) ? intval( $edited_data['id'] ) : 0;
        if ( ! $property_id ) {
            ob_end_clean();
            wp_send_json_error( [ 'message' => 'ID de propiedad inválido' ] );
            return;
        }

        // Verificar usuario local por email
        $local_user_id = self::find_user_by_email( $edited_data['author_email'] ?? '' );
        if ( ! $local_user_id ) {
            ob_end_clean();
            wp_send_json_error( [ 'message' => 'Usuario local no encontrado: ' . ($edited_data['author_email'] ?? 'email vacío') ] );
            return;
        }

        // Verificar si ya está importada
        if ( self::is_property_already_imported( $edited_data['title'] ?? '', $edited_data['author_email'] ?? '' ) ) {
            if ( ! $allow_duplicate ) {
                while (ob_get_level()) { @ob_end_clean(); }
                wp_send_json_error( [ 'message' => 'Esta propiedad ya fue importada' ] );
                return;
            }
        }

        try {
            // Configurar límites para evitar timeouts
            self::set_import_limits();
            // Registrar shutdown handler para capturar errores fatales y evitar respuesta en blanco
            register_shutdown_function(function() {
                $error = error_get_last();
                if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR])) {
                    while (ob_get_level()) { @ob_end_clean(); }
                    if (!headers_sent()) { header('Content-Type: application/json; charset=utf-8'); }
                    echo json_encode([
                        'success' => false,
                        'data' => [
                            'message' => 'Error fatal durante importación: ' . $error['message'],
                            'error_code' => 'fatal_error',
                            'error_details' => [ 'file' => basename($error['file']), 'line' => $error['line'] ]
                        ]
                    ]);
                    exit;
                }
            });

            // Crear el post con datos editados
            $post_data = [
                'post_title'   => sanitize_text_field( $edited_data['title'] ?? '' ),
                'post_content' => wp_kses_post( $edited_data['content'] ?? '' ),
                'post_excerpt' => sanitize_textarea_field( $edited_data['excerpt'] ?? '' ),
                'post_status'  => sanitize_text_field( $edited_data['status'] ?? 'publish' ),
                'post_author'  => $local_user_id,
                // Usar el CPT correcto para este plugin
                'post_type'    => 'properties',
                'meta_input'   => [
                    '_bdweb_original_id' => $property_id,
                    '_bdweb_imported_at' => current_time( 'mysql' ),
                    
                    // Meta campos editados
                    // Precio: guardar en ambas claves para compatibilidad (tema y Crocoblock)
                    '_price' => sanitize_text_field( $edited_data['price'] ?? '' ),
                    'precio' => isset($edited_data['price']) ? floatval( $edited_data['price'] ) : '',
                    'comision' => sanitize_text_field( $edited_data['comision'] ?? '' ),
                    'condiciones_para_comision' => sanitize_textarea_field( $edited_data['condiciones_para_comision'] ?? '' ),
                    '_bedrooms' => sanitize_text_field( $edited_data['bedrooms'] ?? '' ),
                    '_bathrooms' => sanitize_text_field( $edited_data['bathrooms'] ?? '' ),
                    '_property_size' => sanitize_text_field( $edited_data['property_size'] ?? '' ),
                    '_area_size' => sanitize_text_field( $edited_data['area_size'] ?? '' ),
                    '_year_built' => sanitize_text_field( $edited_data['year_built'] ?? '' ),
                    'video' => esc_url_raw( $edited_data['video'] ?? '' ),
                    // Compat: guardar video en claves alternativas que usa el otro plugin/tema
                    'visita_virtual' => esc_url_raw( $edited_data['video'] ?? '' ),
                    '_video-tour-url' => esc_url_raw( $edited_data['video'] ?? '' ),
                    // Campos adicionales solicitados
                    'techado' => sanitize_text_field( $edited_data['techado'] ?? '' ),
                    'patio_maniobras' => sanitize_text_field( $edited_data['patio_maniobras'] ?? '' ),
                    'frente' => sanitize_text_field( $edited_data['frente'] ?? '' ),
                    'fondo' => sanitize_text_field( $edited_data['fondo'] ?? '' ),
                    'uso_de_suelo' => sanitize_text_field( $edited_data['uso_suelo'] ?? '' ),
                    
                    // Ubicación editada (guardar claves usadas por el tema y compatibilidad del otro plugin)
                    '_address' => sanitize_text_field( $edited_data['address'] ?? '' ),
                    '_latitude' => sanitize_text_field( $edited_data['latitude'] ?? '' ),
                    '_longitude' => sanitize_text_field( $edited_data['longitude'] ?? '' ),
                    '_codigo_postal' => sanitize_text_field( $edited_data['codigo_postal'] ?? '' ),
                    // Claves visibles en el metabox del tema
                    'direccion' => sanitize_text_field( $edited_data['address'] ?? '' ),
                    'latitud' => sanitize_text_field( $edited_data['latitude'] ?? '' ),
                    'longitud' => sanitize_text_field( $edited_data['longitude'] ?? '' ),
                    'codigo_postal' => sanitize_text_field( $edited_data['codigo_postal'] ?? '' ),
                ]
            ];

            $post_id = wp_insert_post( $post_data );

            if ( is_wp_error( $post_id ) ) {
                throw new \Exception( 'Error creando post: ' . $post_id->get_error_message() );
            }

            // Procesar taxonomías editadas (intentar ambos slugs para compatibilidad)
            if ( !empty( $edited_data['colonia'] ) ) {
                self::assign_or_create_term( $post_id, sanitize_text_field($edited_data['colonia']), 'property_colonia' );
                self::assign_or_create_term( $post_id, sanitize_text_field($edited_data['colonia']), 'colonia' );
            }

            if ( !empty( $edited_data['ciudad'] ) ) {
                self::assign_or_create_term( $post_id, sanitize_text_field($edited_data['ciudad']), 'property_ciudad' );
                self::assign_or_create_term( $post_id, sanitize_text_field($edited_data['ciudad']), 'ciudad' );
            }

            if ( !empty( $edited_data['estado'] ) ) {
                self::assign_or_create_term( $post_id, sanitize_text_field($edited_data['estado']), 'property_estado' );
                self::assign_or_create_term( $post_id, sanitize_text_field($edited_data['estado']), 'estados' );
            }

            // Transacción (purpose -> transaccion)
            if ( !empty( $edited_data['transaccion'] ) ) {
                $items = $edited_data['transaccion'];
                if ( is_string( $items ) ) {
                    $items = array_filter( array_map( 'trim', explode( ',', $items ) ) );
                }
                if ( is_array( $items ) ) {
                    foreach ( $items as $name ) {
                        if ( !empty( $name ) ) {
                            self::assign_or_create_term( $post_id, sanitize_text_field( $name ), 'transaccion' );
                        }
                    }
                }
            }

            // Tipo de propiedad (property-type)
            if ( !empty( $edited_data['property_type'] ) ) {
                $items = $edited_data['property_type'];
                if ( is_string( $items ) ) {
                    $items = array_filter( array_map( 'trim', explode( ',', $items ) ) );
                }
                if ( is_array( $items ) ) {
                    foreach ( $items as $name ) {
                        if ( !empty( $name ) ) {
                            self::assign_or_create_term( $post_id, sanitize_text_field( $name ), 'property-type' );
                        }
                    }
                }
            }

            // Amenidades (como términos)
            if ( !empty( $edited_data['amenidades'] ) ) {
                $amenidades = $edited_data['amenidades'];
                if ( is_string( $amenidades ) ) {
                    $amenidades = array_filter( array_map( 'trim', explode( ',', $amenidades ) ) );
                }
                if ( is_array( $amenidades ) ) {
                    foreach ( $amenidades as $amenity_name ) {
                        if ( !empty( $amenity_name ) ) {
                            self::assign_or_create_term( $post_id, sanitize_text_field( $amenity_name ), 'amenidades' );
                        }
                    }
                    // También guardar lista en meta para compatibilidad (como en import normal)
                    update_post_meta( $post_id, '_amenities', $amenidades );
                }
            }

            // Geocoding si no hay coordenadas pero hay dirección
            if ( empty( $edited_data['latitude'] ) && empty( $edited_data['longitude'] ) && !empty( $edited_data['address'] ) ) {
                $full_address = trim( $edited_data['address'] . ', ' . $edited_data['colonia'] . ', ' . $edited_data['ciudad'] . ', ' . $edited_data['estado'] );
                self::geocode_and_save( $post_id, $full_address );
            }

            // Procesar imágenes si están disponibles
            if ( !empty( $edited_data['featured_media'] ) || !empty( $edited_data['gallery'] ) ) {
                self::send_heartbeat( 'Descargando imágenes...' );
                self::process_property_images( $post_id, $edited_data['featured_media'] ?? '', $edited_data['gallery'] ?? [] );
            }

            // Limpiar buffers previos antes de enviar JSON
            while (ob_get_level()) { @ob_end_clean(); }
            wp_send_json_success( [
                'message' => 'Propiedad importada exitosamente con datos editados',
                'post_id' => $post_id,
                'post_type' => get_post_type( $post_id ),
                'edit_link' => admin_url( 'post.php?post=' . $post_id . '&action=edit' ),
                'view_link' => get_permalink( $post_id )
            ] );

    } catch ( \Exception $e ) {
            while (ob_get_level()) { @ob_end_clean(); }
            wp_send_json_error( [ 'message' => 'Error en importación: ' . $e->getMessage() ] );
        }
    }

    public static function debug_property() : void {
        // Verificaciones básicas
        if ( ! current_user_can( HPBDWEB_CAP ) ) {
            wp_send_json_error( [ 'message' => 'Permisos insuficientes.' ] );
            return;
        }

        check_ajax_referer( HPBDWEB_NONCE );

        $property_id = isset( $_POST['property_id'] ) ? intval( $_POST['property_id'] ) : 0;

        if ( ! $property_id ) {
            wp_send_json_error( [ 'message' => 'ID de propiedad requerido' ] );
            return;
        }

    // Obtener datos reales de la API (incluir 'content' y 'amenidades' con _fields)
    $response = wp_remote_get( self::BDWEB_API_BASE . '/' . $property_id . '?_fields=id,title,content,excerpt,status,autor,meta,featured_media,colonia,ciudad,estados,amenidades,date' );
        
        if ( is_wp_error( $response ) ) {
            wp_send_json_error( [ 'message' => 'Error API: ' . $response->get_error_message() ] );
            return;
        }

        $body = wp_remote_retrieve_body( $response );
        $property_data = json_decode( $body, true );

        if ( json_last_error() !== JSON_ERROR_NONE ) {
            wp_send_json_error( [ 'message' => 'Error JSON: ' . json_last_error_msg() ] );
            return;
        }

        // Extraer información del autor
        $author_email = isset($property_data['autor']['correo']) ? $property_data['autor']['correo'] : '';
        $author_name = isset($property_data['autor']['nombre']) ? $property_data['autor']['nombre'] : '';
        $property_title = isset($property_data['title']['rendered']) ? $property_data['title']['rendered'] : '';
        
        // Verificar usuario local
        $local_user_id = self::find_user_by_email( $author_email );
        $local_user = $local_user_id ? get_userdata( $local_user_id ) : null;
        
        // Verificar si ya está importada
        $already_imported = self::is_property_already_imported( $property_title, $author_email );

        // Extraer taxonomías
        $colonia_nombre = !empty($property_data['colonia']) ? $property_data['colonia'][0]['nombre'] : 'Sin colonia';
        $ciudad_nombre = !empty($property_data['ciudad']) ? $property_data['ciudad'][0]['nombre'] : 'Sin ciudad';
        $estado_nombre = !empty($property_data['estados']) ? $property_data['estados'][0]['nombre'] : 'Sin estado';
        
        // Información de ubicación
        $address = isset($property_data['meta']['address']) ? $property_data['meta']['address'] : 'Sin dirección';
        $lat = isset($property_data['meta']['latitud']) ? $property_data['meta']['latitud'] : '';
        $lng = isset($property_data['meta']['longitud']) ? $property_data['meta']['longitud'] : '';
        $codigo_postal = isset($property_data['meta']['codigo']) ? $property_data['meta']['codigo'] : '';
        $has_coordinates = !empty($lat) && !empty($lng);

        // Extraer detalles de la propiedad
        $price = isset($property_data['meta']['price']) ? '$' . number_format($property_data['meta']['price']) : 'Sin precio';
        $bedrooms = isset($property_data['meta']['bedrooms']) ? $property_data['meta']['bedrooms'] : 'Sin habitaciones';
        $bathrooms = isset($property_data['meta']['bathsrooms']) ? $property_data['meta']['bathsrooms'] : 'Sin baños';
        $property_size = isset($property_data['meta']['property-size']) ? $property_data['meta']['property-size'] : 'Sin tamaño';
        $area_size = isset($property_data['meta']['area-size']) ? $property_data['meta']['area-size'] : 'Sin área';

        // Información de imágenes
        $gallery = isset($property_data['meta']['gallery']) ? $property_data['meta']['gallery'] : [];
        $featured_media = isset($property_data['featured_media']) ? $property_data['featured_media'] : '';

        wp_send_json_success( [
            'debug_info' => [
                'property_id' => $property_id,
                'title' => $property_title,
                'author_email' => $author_email, // Mantener para compatibilidad
                'author_info' => [
                    'name' => $author_name,
                    'email' => $author_email,
                    'phone' => isset($property_data['autor']['telefono']) ? $property_data['autor']['telefono'] : 'Sin teléfono'
                ],
                'local_user_found' => $local_user ? true : false,
                'local_user_info' => $local_user ? [
                    'ID' => $local_user->ID,
                    'display_name' => $local_user->display_name,
                    'user_login' => $local_user->user_login,
                    'user_email' => $local_user->user_email
                ] : 'No encontrado',
                'already_imported' => $already_imported,
                'location_info' => [
                    'address' => $address,
                    'colonia' => $colonia_nombre,
                    'ciudad' => $ciudad_nombre,
                    'estado' => $estado_nombre,
                    'codigo_postal' => $codigo_postal,
                    'latitud' => $lat ?: 'Sin latitud',
                    'longitud' => $lng ?: 'Sin longitud',
                    'has_coordinates' => $has_coordinates,
                    'can_geocode' => !empty($address) && !$has_coordinates,
                    'geocoding_needed' => !$has_coordinates && !empty($address) ? '🗺️ Geocoding automático disponible' : '✅ Coordenadas ya disponibles'
                ],
                'property_details' => [
                    'price' => $price,
                    'bedrooms' => $bedrooms,
                    'bathrooms' => $bathrooms,
                    'property_size' => $property_size,
                    'area_size' => $area_size,
                    'year_built' => isset($property_data['meta']['year-built']) ? $property_data['meta']['year-built'] : 'Sin año'
                ],
                'media_info' => [
                    'featured_media_id' => $featured_media,
                    'gallery_images_count' => count($gallery),
                    'gallery_ids' => array_slice($gallery, 0, 3), // Solo primeras 3
                    'has_images' => !empty($featured_media) || !empty($gallery)
                ],
                'featured_media' => $featured_media, // Para compatibilidad
                'gallery_images' => $gallery, // Para compatibilidad
                'taxonomies_info' => [
                    'colonia_raw' => $property_data['colonia'] ?? [],
                    'ciudad_raw' => $property_data['ciudad'] ?? [],
                    'estado_raw' => $property_data['estados'] ?? []
                ],
                'ready_to_import' => $local_user && !$already_imported ? '✅ Listo para importar' : ($local_user ? '⚠️ Ya importada' : '❌ Usuario no encontrado')
            ]
        ] );
    }

    public static function ajax_test() : void {
        if ( ! current_user_can( HPBDWEB_CAP ) ) {
            wp_send_json_error( [ 'message' => __( 'Permisos insuficientes.', 'hp-bdweb-import' ) ], 403 );
        }

        check_ajax_referer( HPBDWEB_NONCE );

        $payload = [
            'ok'      => true,
            'time'    => current_time( 'mysql' ),
            'site'    => home_url(),
            'version' => HPBDWEB_VER,
            'user'    => wp_get_current_user() ? wp_get_current_user()->user_login : null,
            'msg'     => __( 'AJAX conectado correctamente. Aquí puedes continuar con la lógica de importación.', 'hp-bdweb-import' ),
        ];

        wp_send_json_success( $payload );
    }

    public static function test_api() : void {
        if ( ! current_user_can( HPBDWEB_CAP ) ) {
            wp_send_json_error( [ 'message' => __( 'Permisos insuficientes.', 'hp-bdweb-import' ) ], 403 );
        }

        check_ajax_referer( HPBDWEB_NONCE );

        // First, get the total count using per_page=1 to avoid loading too much data
        $response = wp_remote_get( self::BDWEB_API_BASE . '?per_page=1' );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( [
                'message' => __( 'Error de conexión: ', 'hp-bdweb-import' ) . $response->get_error_message()
            ] );
        }

        $status_code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        if ( $status_code !== 200 ) {
            wp_send_json_error( [
                'message' => sprintf( __( 'Error HTTP %d al conectar con BDweb API', 'hp-bdweb-import' ), $status_code )
            ] );
        }

        $data = json_decode( $body, true );
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            wp_send_json_error( [
                'message' => __( 'Respuesta JSON inválida de la API', 'hp-bdweb-import' )
            ] );
        }

        // Get the total count from response headers
        $headers = wp_remote_retrieve_headers( $response );
        $total_properties = isset( $headers['x-wp-total'] ) ? intval( $headers['x-wp-total'] ) : 0;
        $total_pages = isset( $headers['x-wp-totalpages'] ) ? intval( $headers['x-wp-totalpages'] ) : 0;

        // If headers don't provide total, fallback to counting current response
        if ( ! $total_properties && is_array( $data ) ) {
            $total_properties = count( $data );
        }

        wp_send_json_success( [
            'message' => __( '✓ Conexión exitosa con BDweb API', 'hp-bdweb-import' ),
            'properties_found' => $total_properties,
            'total_pages' => $total_pages,
            'api_url' => self::BDWEB_API_BASE,
            'status_code' => $status_code,
            'per_page_default' => 10
        ] );
    }

    public static function load_properties() : void {
        if ( ! current_user_can( HPBDWEB_CAP ) ) {
            wp_send_json_error( [ 'message' => __( 'Permisos insuficientes.', 'hp-bdweb-import' ) ], 403 );
        }

        check_ajax_referer( HPBDWEB_NONCE );

        $page = isset( $_POST['page'] ) ? max( 1, intval( $_POST['page'] ) ) : 1;
        $per_page = 20; // Reducir para mejor rendimiento

        $response = wp_remote_get( self::BDWEB_API_BASE . "?per_page={$per_page}&page={$page}&orderby=id&order=desc" );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( [
                'message' => __( 'Error al cargar propiedades: ', 'hp-bdweb-import' ) . $response->get_error_message()
            ] );
        }

        $body = wp_remote_retrieve_body( $response );
        $properties = json_decode( $body, true );

        if ( json_last_error() !== JSON_ERROR_NONE || ! is_array( $properties ) ) {
            wp_send_json_error( [
                'message' => __( 'Error al procesar datos de propiedades', 'hp-bdweb-import' )
            ] );
        }

        $formatted_properties = [];
        foreach ( $properties as $property ) {
            $author_email = $property['autor']['correo'] ?? '';
            $property_title = $property['title']['rendered'] ?? __( 'Sin título', 'hp-bdweb-import' );
            $local_user_id = self::find_user_by_email( $author_email );
            $local_user = $local_user_id ? get_userdata( $local_user_id ) : null;

            // Check if already imported
            $already_imported = self::is_property_already_imported( $property_title, $author_email );

            $formatted_properties[] = [
                'id' => $property['id'] ?? 0,
                'title' => $property_title,
                'author_name' => $property['autor']['nombre'] ?? __( 'Desconocido', 'hp-bdweb-import' ),
                'author_email' => $author_email,
                'date' => $property['date'] ?? '',
                'status' => $property['status'] ?? 'unknown',
                'local_user_exists' => $local_user ? true : false,
                'local_user_name' => $local_user ? $local_user->display_name : null,
                'local_user_id' => $local_user_id,
                'already_imported' => $already_imported
            ];
        }

        // Get the total count from headers
        $headers = wp_remote_retrieve_headers( $response );
        $total_properties = isset( $headers['x-wp-total'] ) ? intval( $headers['x-wp-total'] ) : count( $formatted_properties );
        $total_pages = isset( $headers['x-wp-totalpages'] ) ? intval( $headers['x-wp-totalpages'] ) : 1;

        wp_send_json_success( [
            'properties' => $formatted_properties,
            'total' => count( $formatted_properties ),
            'total_available' => $total_properties,
            'current_page' => $page,
            'total_pages' => $total_pages,
            'message' => sprintf( __( 'Página %d de %d: %d propiedades de %d totales', 'hp-bdweb-import' ), $page, $total_pages, count( $formatted_properties ), $total_properties )
        ] );
    }

    public static function preview_property() : void {
        if ( ! current_user_can( HPBDWEB_CAP ) ) {
            wp_send_json_error( [ 'message' => __( 'Permisos insuficientes.', 'hp-bdweb-import' ) ], 403 );
        }

        check_ajax_referer( HPBDWEB_NONCE );

        $property_id = isset( $_POST['property_id'] ) ? intval( $_POST['property_id'] ) : 0;

        if ( ! $property_id ) {
            wp_send_json_error( [ 'message' => __( 'ID de propiedad requerido', 'hp-bdweb-import' ) ] );
        }

    $response = wp_remote_get( self::BDWEB_API_BASE . '/' . $property_id . '?_fields=id,title,content,excerpt,status,autor,meta,featured_media,colonia,ciudad,estados,amenidades,date' );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( [
                'message' => __( 'Error al cargar propiedad: ', 'hp-bdweb-import' ) . $response->get_error_message()
            ] );
        }

        $status_code = wp_remote_retrieve_response_code( $response );
        if ( $status_code === 404 ) {
            wp_send_json_error( [
                'message' => __( 'Propiedad no encontrada', 'hp-bdweb-import' )
            ] );
        }

        $body = wp_remote_retrieve_body( $response );
        $property = json_decode( $body, true );

        if ( json_last_error() !== JSON_ERROR_NONE ) {
            wp_send_json_error( [
                'message' => __( 'Error al procesar datos de la propiedad', 'hp-bdweb-import' )
            ] );
        }

        // Check if already imported by title AND author email
        $property_title = $property['title']['rendered'] ?? '';
        $author_email = $property['autor']['correo'] ?? '';
        
        $already_imported = self::is_property_already_imported( $property_title, $author_email );

        // Check if local user exists
        $author_email = $property['autor']['correo'] ?? '';
        $local_user_id = self::find_user_by_email( $author_email );
        $local_user = $local_user_id ? get_userdata( $local_user_id ) : null;

        wp_send_json_success( [
            'property' => [
                'id' => $property['id'] ?? 0,
                'title' => $property['title']['rendered'] ?? __( 'Sin título', 'hp-bdweb-import' ),
                'content' => wp_trim_words( strip_tags( $property['content']['rendered'] ?? '' ), 20 ),
                'author_name' => $property['autor']['nombre'] ?? __( 'Desconocido', 'hp-bdweb-import' ),
                'author_email' => $author_email,
                'author_phone' => $property['autor']['telefono'] ?? '',
                'price' => $property['meta']['price'] ?? '',
                'address' => $property['meta']['address'] ?? '',
                'bedrooms' => $property['meta']['bedrooms'] ?? '',
                'bathsrooms' => $property['meta']['bathsrooms'] ?? '',
                'property_size' => $property['meta']['property-size'] ?? '',
                'area_size' => $property['meta']['area-size'] ?? '',
                'parking' => $property['meta']['garaje'] ?? '',
                'year_built' => $property['meta']['year-built'] ?? '',
                'virtual_tour' => $property['meta']['virtual-tour'] ?? '',
                'date' => $property['date'] ?? ''
            ],
            'already_imported' => $already_imported,
            'local_user_exists' => $local_user ? true : false,
            'local_user_name' => $local_user ? $local_user->display_name : null,
            'local_user_id' => $local_user_id
        ] );
    }

    public static function import_property() : void {
        // Activar output buffering para capturar cualquier output inesperado
        ob_start();
        
        try {
            error_log( "HPBDWEB: === INICIO IMPORTACIÓN INDIVIDUAL ===" );
            
            if ( ! current_user_can( HPBDWEB_CAP ) ) {
                ob_end_clean();
                wp_send_json_error( [ 'message' => 'Permisos insuficientes.' ], 403 );
                return;
            }

            check_ajax_referer( HPBDWEB_NONCE );
            
            error_log( "HPBDWEB: Verificaciones de seguridad pasadas" );
        
        // Configurar límites de tiempo y memoria para evitar timeouts
        self::set_import_limits();
        
        // Configurar handler para errores fatales
        register_shutdown_function(function() {
            $error = error_get_last();
            error_log("HPBDWEB: Shutdown function ejecutada. Error: " . print_r($error, true));
            
            if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR])) {
                error_log("HPBDWEB: Error fatal detectado: {$error['message']} en {$error['file']}:{$error['line']}");
                
                // Limpiar cualquier output y enviar error JSON
                while (ob_get_level()) {
                    ob_end_clean();
                }
                
                // Asegurar headers correctos
                if (!headers_sent()) {
                    header('Content-Type: application/json; charset=utf-8');
                }
                
                echo json_encode([
                    'success' => false,
                    'data' => [
                        'message' => 'Error fatal: ' . $error['message'],
                        'error_code' => 'fatal_error',
                        'error_details' => [
                            'file' => basename($error['file']),
                            'line' => $error['line'],
                            'type' => 'Fatal Error'
                        ]
                    ]
                ]);
                exit;
            }
        });        $property_id = isset( $_POST['property_id'] ) ? intval( $_POST['property_id'] ) : 0;
        error_log( "HPBDWEB: Property ID recibido: " . ($_POST['property_id'] ?? 'null') . " -> parseado: $property_id" );

        if ( ! $property_id ) {
            error_log( "HPBDWEB: ID de propiedad inválido" );
            ob_end_clean();
            wp_send_json_error( [ 'message' => 'ID de propiedad requerido' ] );
            return;
        }            // Get property data first to check title
            error_log( "HPBDWEB: Solicitando propiedad ID $property_id desde " . self::BDWEB_API_BASE );
            
            $response = wp_remote_get( self::BDWEB_API_BASE . '/' . $property_id, [
                'timeout' => 30,
                'headers' => [
                    'User-Agent' => 'WordPress/' . get_bloginfo( 'version' ) . '; ' . home_url()
                ]
            ]);
            
            if ( is_wp_error( $response ) ) {
                $error_msg = $response->get_error_message();
                error_log( "HPBDWEB: Error en API request: $error_msg" );
                wp_send_json_error( [ 'message' => __( 'Error al obtener datos de la propiedad: ', 'hp-bdweb-import' ) . $error_msg ] );
            }

            $http_code = wp_remote_retrieve_response_code( $response );
            if ( $http_code !== 200 ) {
                error_log( "HPBDWEB: HTTP error code: $http_code" );
                wp_send_json_error( [ 'message' => sprintf( __( 'Error HTTP %d al obtener la propiedad', 'hp-bdweb-import' ), $http_code ) ] );
            }

            $body = wp_remote_retrieve_body( $response );
            if ( empty( $body ) ) {
                error_log( "HPBDWEB: Respuesta vacía de la API" );
                wp_send_json_error( [ 'message' => __( 'Respuesta vacía de la API', 'hp-bdweb-import' ) ] );
            }

            $bdweb_property = json_decode( $body, true );
            if ( json_last_error() !== JSON_ERROR_NONE ) {
                $json_error = json_last_error_msg();
                error_log( "HPBDWEB: Error JSON: $json_error" );
                wp_send_json_error( [ 'message' => __( 'Error al decodificar JSON: ', 'hp-bdweb-import' ) . $json_error ] );
            }

            if ( empty( $bdweb_property ) ) {
                error_log( "HPBDWEB: Datos de propiedad vacíos después de decodificar JSON" );
                wp_send_json_error( [ 'message' => __( 'No se encontraron datos para esta propiedad', 'hp-bdweb-import' ) ] );
            }

            $property_title = $bdweb_property['title']['rendered'] ?? '';
            if ( empty( $property_title ) ) {
                error_log( "HPBDWEB: Título de propiedad vacío" );
                wp_send_json_error( [ 'message' => __( 'La propiedad no tiene título', 'hp-bdweb-import' ) ] );
            }

            // Check if already imported by title AND author email
            $author_email = $bdweb_property['autor']['correo'] ?? '';
            
            if ( self::is_property_already_imported( $property_title, $author_email ) ) {
                wp_send_json_error( [
                    'message' => sprintf( __( 'La propiedad "%s" del autor %s ya fue importada', 'hp-bdweb-import' ), $property_title, $author_email )
                ] );
            }

            error_log( "HPBDWEB: Iniciando process_property_import para ID $property_id" );
            error_log( "HPBDWEB: Memoria antes de process_property_import: " . memory_get_usage(true) . " bytes" );
            
            // ACTIVANDO IMPORT REAL CON DEBUG
            error_log( "HPBDWEB: Iniciando import real con debugging granular" );
            
            $result = self::process_property_import( $property_id, $bdweb_property );
            
            error_log( "HPBDWEB: Import real completado exitosamente para ID $property_id" );

            if ( is_wp_error( $result ) ) {
                $error_msg = $result->get_error_message();
                $error_code = $result->get_error_code();
                error_log( "HPBDWEB: Error en process_property_import ($error_code): $error_msg" );
                wp_send_json_error( [ 
                    'message' => "Error detallado ($error_code): " . $error_msg,
                    'error_code' => $error_code,
                    'error_details' => $result->get_error_data()
                ] );
            }

            error_log( "HPBDWEB: Importación exitosa para ID $property_id" );
            
            // Limpiar cualquier output inesperado antes de enviar JSON
            ob_end_clean();
            wp_send_json_success( $result );
            
        } catch ( Throwable $e ) {
            // Capturar cualquier output que haya ocurrido
            $unexpected_output = ob_get_contents();
            ob_end_clean();
            
            $error_msg = $e->getMessage();
            $file = $e->getFile();
            $line = $e->getLine();
            
            error_log( "HPBDWEB: Error capturado en import_property: $error_msg en $file:$line" );
            error_log( "HPBDWEB: Stack trace: " . $e->getTraceAsString() );
            
            if ( $unexpected_output ) {
                error_log( "HPBDWEB: Output inesperado capturado: " . $unexpected_output );
            }
            
            wp_send_json_error( [ 
                'message' => 'Error detallado: ' . $error_msg,
                'error_code' => get_class($e),
                'error_details' => [
                    'file' => basename($file),
                    'line' => $line,
                    'message' => $error_msg,
                    'type' => get_class($e),
                    'unexpected_output' => $unexpected_output
                ]
            ] );
        }
    }

    public static function import_all() : void {
        if ( ! current_user_can( HPBDWEB_CAP ) ) {
            wp_send_json_error( [ 'message' => __( 'Permisos insuficientes.', 'hp-bdweb-import' ) ], 403 );
        }

        check_ajax_referer( HPBDWEB_NONCE );
        
        // Configurar límites de tiempo y memoria para evitar timeouts
        self::set_import_limits();

        // Get properties per page (we'll do it in batches)
        $per_page = 50;
        $page = isset( $_POST['page'] ) ? max( 1, intval( $_POST['page'] ) ) : 1;

        $response = wp_remote_get( self::BDWEB_API_BASE . "?per_page={$per_page}&page={$page}&orderby=id&order=desc" );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( [
                'message' => __( 'Error al cargar propiedades: ', 'hp-bdweb-import' ) . $response->get_error_message()
            ] );
        }

        $body = wp_remote_retrieve_body( $response );
        $properties = json_decode( $body, true );

        if ( json_last_error() !== JSON_ERROR_NONE || ! is_array( $properties ) ) {
            wp_send_json_error( [
                'message' => __( 'Error al procesar datos de propiedades', 'hp-bdweb-import' )
            ] );
        }

        $results = [
            'imported' => 0,
            'skipped' => 0,
            'errors' => 0,
            'details' => [],
            'processed_ids' => []
        ];

        foreach ( $properties as $property ) {
            // Enviar heartbeat cada propiedad para mantener conexión viva
            self::send_heartbeat();
            
            $property_id = $property['id'] ?? 0;
            
            if ( ! $property_id ) {
                $results['errors']++;
                $results['details'][] = __( 'ID de propiedad inválido', 'hp-bdweb-import' );
                continue;
            }

            // Check if already imported by title AND author email
            $property_title = $property['title']['rendered'] ?? '';
            $author_email = $property['autor']['correo'] ?? '';

            if ( self::is_property_already_imported( $property_title, $author_email ) ) {
                $results['skipped']++;
                $results['details'][] = sprintf( __( 'ID %d: Ya importada (%s - %s)', 'hp-bdweb-import' ), $property_id, $property_title, $author_email );
                $results['processed_ids'][] = $property_id;
                continue;
            }

            // Check if local user exists
            $author_email = $property['autor']['correo'] ?? '';
            $local_user_id = self::find_user_by_email( $author_email );
            
            if ( ! $local_user_id ) {
                $results['skipped']++;
                $results['details'][] = sprintf( __( 'ID %d: Sin usuario local (%s)', 'hp-bdweb-import' ), $property_id, $author_email );
                $results['processed_ids'][] = $property_id;
                continue;
            }

            // Try to import
            $import_result = self::process_property_import( $property_id, $property );
            
            if ( is_wp_error( $import_result ) ) {
                $results['errors']++;
                $results['details'][] = sprintf( __( 'ID %d: Error - %s', 'hp-bdweb-import' ), $property_id, $import_result->get_error_message() );
            } else {
                $results['imported']++;
                $user_name = get_userdata( $local_user_id )->display_name ?? '';
                $results['details'][] = sprintf( __( 'ID %d: Importada → %s', 'hp-bdweb-import' ), $property_id, $user_name );
            }
            
            $results['processed_ids'][] = $property_id;
        }

        // Check if there are more pages
        $headers = wp_remote_retrieve_headers( $response );
        $total_pages = isset( $headers['x-wp-totalpages'] ) ? intval( $headers['x-wp-totalpages'] ) : 1;
        $has_more_pages = $page < $total_pages;

        wp_send_json_success( [
            'results' => $results,
            'current_page' => $page,
            'total_pages' => $total_pages,
            'has_more_pages' => $has_more_pages,
            'next_page' => $has_more_pages ? $page + 1 : null,
            'message' => sprintf( 
                __( 'Página %d/%d: %d importadas, %d omitidas, %d errores', 'hp-bdweb-import' ),
                $page, $total_pages, $results['imported'], $results['skipped'], $results['errors']
            )
        ] );
    }

    public static function import_page() : void {
        if ( ! current_user_can( HPBDWEB_CAP ) ) {
            wp_send_json_error( [ 'message' => __( 'Permisos insuficientes.', 'hp-bdweb-import' ) ], 403 );
        }

        check_ajax_referer( HPBDWEB_NONCE );
        
        // Configurar límites de tiempo y memoria para evitar timeouts
        self::set_import_limits();

        $page = isset( $_POST['page'] ) ? max( 1, intval( $_POST['page'] ) ) : 1;
        $per_page = 20; // Menos propiedades por página para evitar timeout

        $response = wp_remote_get( self::BDWEB_API_BASE . "?per_page={$per_page}&page={$page}&orderby=id&order=desc" );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( [
                'message' => __( 'Error al cargar propiedades: ', 'hp-bdweb-import' ) . $response->get_error_message()
            ] );
        }

        $body = wp_remote_retrieve_body( $response );
        $properties = json_decode( $body, true );

        if ( json_last_error() !== JSON_ERROR_NONE || ! is_array( $properties ) ) {
            wp_send_json_error( [
                'message' => __( 'Error al procesar datos de propiedades', 'hp-bdweb-import' )
            ] );
        }

        $results = [
            'imported' => 0,
            'skipped' => 0,
            'errors' => 0,
            'details' => []
        ];

        foreach ( $properties as $property ) {
            // Enviar heartbeat cada propiedad para mantener conexión viva
            self::send_heartbeat();
            
            $property_id = $property['id'] ?? 0;
            $property_title = $property['title']['rendered'] ?? '';
            
            if ( ! $property_id ) {
                $results['errors']++;
                $results['details'][] = __( 'ID de propiedad inválido', 'hp-bdweb-import' );
                continue;
            }

            // Check if already imported by title AND author email
            $author_email = $property['autor']['correo'] ?? '';
            
            if ( self::is_property_already_imported( $property_title, $author_email ) ) {
                $results['skipped']++;
                $results['details'][] = sprintf( __( '"%s": Ya importada (%s)', 'hp-bdweb-import' ), $property_title, $author_email );
                continue;
            }

            // Check if local user exists
            $author_email = $property['autor']['correo'] ?? '';
            $local_user_id = self::find_user_by_email( $author_email );
            
            if ( ! $local_user_id ) {
                $results['skipped']++;
                $results['details'][] = sprintf( __( '"%s": Sin usuario local (%s)', 'hp-bdweb-import' ), $property_title, $author_email );
                continue;
            }

            // Try to import (with images - now optimized for no timeout)
            $import_result = self::process_property_import( $property_id, $property, true ); // true = with images
            
            if ( is_wp_error( $import_result ) ) {
                $results['errors']++;
                $results['details'][] = sprintf( __( '"%s": Error - %s', 'hp-bdweb-import' ), $property_title, $import_result->get_error_message() );
            } else {
                $results['imported']++;
                $user_name = get_userdata( $local_user_id )->display_name ?? '';
                $results['details'][] = sprintf( __( '"%s": Importada → %s', 'hp-bdweb-import' ), $property_title, $user_name );
            }
        }

        wp_send_json_success( [
            'results' => $results,
            'message' => sprintf( 
                __( 'Página %d procesada: %d importadas, %d omitidas, %d errores', 'hp-bdweb-import' ),
                $page, $results['imported'], $results['skipped'], $results['errors']
            )
        ] );
    }

    public static function batch_import() : void {
        if ( ! current_user_can( HPBDWEB_CAP ) ) {
            wp_send_json_error( [ 'message' => __( 'Permisos insuficientes.', 'hp-bdweb-import' ) ], 403 );
        }

        check_ajax_referer( HPBDWEB_NONCE );
        
        // Configurar límites de tiempo y memoria para evitar timeouts
        self::set_import_limits();

        $offset = isset( $_POST['offset'] ) ? max( 0, intval( $_POST['offset'] ) ) : 0;
        $batch_size = 10; // Lotes de 10 propiedades
        
        // Calculate page number for API (BDweb uses 1-based pagination)
        $page = floor( $offset / $batch_size ) + 1;
        $per_page = $batch_size;

        $response = wp_remote_get( self::BDWEB_API_BASE . "?per_page={$per_page}&page={$page}&orderby=id&order=desc" );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( [
                'message' => __( 'Error al cargar propiedades: ', 'hp-bdweb-import' ) . $response->get_error_message()
            ] );
        }

        $body = wp_remote_retrieve_body( $response );
        $properties = json_decode( $body, true );

        if ( json_last_error() !== JSON_ERROR_NONE || ! is_array( $properties ) ) {
            wp_send_json_error( [
                'message' => __( 'Error al procesar datos de propiedades', 'hp-bdweb-import' )
            ] );
        }

        // Get total count
        $headers = wp_remote_retrieve_headers( $response );
        $total_properties = isset( $headers['x-wp-total'] ) ? intval( $headers['x-wp-total'] ) : 0;

        $results = [
            'imported' => 0,
            'skipped' => 0,
            'errors' => 0,
            'details' => [],
            'processed_ids' => []
        ];

        $batch_start = $offset + 1;
        $batch_end = min( $offset + count( $properties ), $total_properties );

        foreach ( $properties as $property ) {
            // Enviar heartbeat cada propiedad para mantener conexión viva
            self::send_heartbeat();
            
            $property_id = $property['id'] ?? 0;
            $property_title = $property['title']['rendered'] ?? '';
            
            if ( ! $property_id ) {
                $results['errors']++;
                $results['details'][] = __( 'ID de propiedad inválido', 'hp-bdweb-import' );
                continue;
            }

            // Check if already imported by title AND author email
            $author_email = $property['autor']['correo'] ?? '';
            
            if ( self::is_property_already_imported( $property_title, $author_email ) ) {
                $results['skipped']++;
                $results['details'][] = sprintf( __( '"%s": Ya importada (%s)', 'hp-bdweb-import' ), $property_title, $author_email );
                $results['processed_ids'][] = $property_id;
                continue;
            }

            // Check if local user exists
            $author_email = $property['autor']['correo'] ?? '';
            $local_user_id = self::find_user_by_email( $author_email );
            
            if ( ! $local_user_id ) {
                $results['skipped']++;
                $results['details'][] = sprintf( __( '"%s": Sin usuario local (%s)', 'hp-bdweb-import' ), $property_title, $author_email );
                $results['processed_ids'][] = $property_id;
                continue;
            }

            // Try to import (with images - now optimized for no timeout)
            $import_result = self::process_property_import( $property_id, $property, true ); // true = with images
            
            if ( is_wp_error( $import_result ) ) {
                $results['errors']++;
                $results['details'][] = sprintf( __( '"%s": Error - %s', 'hp-bdweb-import' ), $property_title, $import_result->get_error_message() );
            } else {
                $results['imported']++;
                $user_name = get_userdata( $local_user_id )->display_name ?? '';
                $results['details'][] = sprintf( __( '"%s": Importada → %s', 'hp-bdweb-import' ), $property_title, $user_name );
            }
            
            $results['processed_ids'][] = $property_id;
        }

        $next_offset = $offset + count( $properties );
        $has_more = $next_offset < $total_properties;

        wp_send_json_success( [
            'results' => $results,
            'batch_info' => [
                'start' => $batch_start,
                'end' => $batch_end,
                'total' => $total_properties,
                'processed' => count( $properties )
            ],
            'pagination' => [
                'offset' => $offset,
                'next_offset' => $next_offset,
                'has_more' => $has_more,
                'progress_percent' => round( ( $next_offset / $total_properties ) * 100, 1 )
            ],
            'message' => sprintf( 
                __( 'Lote %d-%d de %d: %d importadas, %d omitidas, %d errores', 'hp-bdweb-import' ),
                $batch_start, $batch_end, $total_properties, $results['imported'], $results['skipped'], $results['errors']
            )
        ] );
    }

    private static function process_property_import( $property_id, $bdweb_property = null, $import_images = true ) {
        try {
            error_log( "HPBDWEB: === INICIO process_property_import ===" );
            error_log( "HPBDWEB: Property ID: $property_id" );
            error_log( "HPBDWEB: Import images: " . ($import_images ? 'true' : 'false') );
            error_log( "HPBDWEB: BDweb property provided: " . (is_null($bdweb_property) ? 'false' : 'true') );
            
            // Get property data from BDweb API if not provided
            if ( ! $bdweb_property ) {
                error_log( "HPBDWEB: Obteniendo datos de propiedad desde API" );
                $response = wp_remote_get( self::BDWEB_API_BASE . '/' . $property_id, [
                    'timeout' => 30
                ] );

                error_log( "HPBDWEB: Response obtenido de API" );

                if ( is_wp_error( $response ) ) {
                    error_log( "HPBDWEB: Error en wp_remote_get: " . $response->get_error_message() );
                    return new WP_Error( 'api_error', __( 'Error al obtener datos de la propiedad', 'hp-bdweb-import' ) );
                }

                error_log( "HPBDWEB: Extrayendo body del response" );
                $body = wp_remote_retrieve_body( $response );
                error_log( "HPBDWEB: Body extraído, tamaño: " . strlen($body) . " chars" );
                
                $bdweb_property = json_decode( $body, true );
                error_log( "HPBDWEB: JSON decodificado" );

                if ( json_last_error() !== JSON_ERROR_NONE ) {
                    error_log( "HPBDWEB: Error en JSON decode: " . json_last_error_msg() );
                    return new WP_Error( 'json_error', __( 'Error al procesar JSON de la propiedad', 'hp-bdweb-import' ) );
                }
                error_log( "HPBDWEB: Propiedad obtenida correctamente de API" );
            }

        // Debug: Log data structure
        error_log( "HPBDWEB: Estructura de bdweb_property: " . print_r( array_keys( $bdweb_property ), true ) );
        
        // Find local user by author email - try different possible structures
        $author_email = '';
        if ( isset( $bdweb_property['autor']['correo'] ) ) {
            $author_email = $bdweb_property['autor']['correo'];
            error_log( "HPBDWEB: Email encontrado en autor->correo: $author_email" );
        } elseif ( isset( $bdweb_property['acf']['correo_del_autor'] ) ) {
            $author_email = $bdweb_property['acf']['correo_del_autor'];
            error_log( "HPBDWEB: Email encontrado en acf->correo_del_autor: $author_email" );
        } elseif ( isset( $bdweb_property['meta']['author_email'] ) ) {
            $author_email = $bdweb_property['meta']['author_email'];
            error_log( "HPBDWEB: Email encontrado en meta->author_email: $author_email" );
        } else {
            error_log( "HPBDWEB: No se encontró email del autor en ninguna estructura conocida" );
            if ( isset( $bdweb_property['autor'] ) ) {
                error_log( "HPBDWEB: Estructura de autor disponible: " . print_r( $bdweb_property['autor'], true ) );
            }
            if ( isset( $bdweb_property['acf'] ) ) {
                error_log( "HPBDWEB: Campos ACF disponibles: " . print_r( array_keys( $bdweb_property['acf'] ), true ) );
            }
            if ( isset( $bdweb_property['meta'] ) ) {
                error_log( "HPBDWEB: Campos meta disponibles: " . print_r( array_keys( $bdweb_property['meta'] ), true ) );
            }
        }
        
        if ( empty( $author_email ) ) {
            return new WP_Error( 'no_author_email', __( 'No se encontró el email del autor en los datos de la propiedad', 'hp-bdweb-import' ) );
        }
        
        $local_user_id = self::find_user_by_email( $author_email );
        
        if ( ! $local_user_id ) {
            return new WP_Error( 'user_not_found', sprintf( 
                __( 'No se encontró un usuario local con el email: %s', 'hp-bdweb-import' ),
                $author_email
            ) );
        }
        
        error_log( "HPBDWEB: Usuario local encontrado ID: $local_user_id para email: $author_email" );

        error_log( "HPBDWEB: Iniciando mapeo de datos de propiedad" );
        // Map BDweb data to local structure
        $post_data = [
            'post_type' => 'properties',
            'post_title' => $bdweb_property['title']['rendered'] ?? __( 'Propiedad importada', 'hp-bdweb-import' ),
            'post_content' => $bdweb_property['content']['rendered'] ?? '',
            'post_status' => 'publish',
            'post_author' => $local_user_id, // Assign to local user
            'meta_input' => [
                '_bdweb_property_id' => $property_id,
                '_bdweb_import_date' => current_time( 'mysql' ),
                // Map meta fields from BDweb to local structure (usando estructura correcta)
                'precio' => $bdweb_property['meta']['price'] ?? '',
                'periodo_pago' => $bdweb_property['meta']['payment-period'] ?? '',
                'deposito' => $bdweb_property['meta']['deposit'] ?? '',
                'tamano_de_construccion' => $bdweb_property['meta']['property-size'] ?? '',
                'metros_terreno' => $bdweb_property['meta']['area-size'] ?? '',
                'comision' => $bdweb_property['meta']['comision'] ?? '',
                'condiciones_para_comision' => $bdweb_property['meta']['condiciones_para_comision'] ?? '',
                'habitaciones' => $bdweb_property['meta']['bedrooms'] ?? '',
                'banos' => $bdweb_property['meta']['bathsrooms'] ?? '',
                'numero_estacionamientos' => $bdweb_property['meta']['garaje'] ?? '',
                'tipo_estacionamiento' => $bdweb_property['meta']['parking'] ?? '',
                'ano_construccion' => $bdweb_property['meta']['year-built'] ?? '',
                'mascotas_permitidas' => $bdweb_property['meta']['allowed-pets'] ?? '',
                'amueblado' => $bdweb_property['meta']['amueblado'] ?? '',
                'tipo_terreno' => $bdweb_property['meta']['terreno'] ?? '',
                'techado' => $bdweb_property['meta']['techado'] ?? '',
                'patio_maniobras' => $bdweb_property['meta']['pation_maniobras'] ?? '',
                'frente' => $bdweb_property['meta']['frente'] ?? '',
                'fondo' => $bdweb_property['meta']['fondo'] ?? '',
                'altura_techo' => $bdweb_property['meta']['ceiling-height'] ?? '',
                'uso_de_suelo' => $bdweb_property['meta']['uso_suelo'] ?? '',
                'luz' => $bdweb_property['meta']['luz'] ?? '',
                'direccion' => $bdweb_property['meta']['address'] ?? '',
                '_address' => $bdweb_property['meta']['address'] ?? '', // Campo requerido para el plugin crear-propiedad
                'codigo_postal' => $bdweb_property['meta']['codigo'] ?? '',
                'pisos_totales' => $bdweb_property['meta']['total-floors'] ?? '',
                'piso' => $bdweb_property['meta']['floor'] ?? '',
                'visita_virtual' => $bdweb_property['meta']['virtual-tour'] ?? '',
                'video' => $bdweb_property['meta']['video'] ?? '',
                'latitud' => $bdweb_property['meta']['latitud'] ?? '',
                'longitud' => $bdweb_property['meta']['longitud'] ?? '',
                // Campos adicionales para compatibilidad con crear-propiedad
                '_latitude' => $bdweb_property['meta']['latitud'] ?? '',
                '_longitude' => $bdweb_property['meta']['longitud'] ?? '',
                '_price' => $bdweb_property['meta']['price'] ?? '',
                '_bedrooms' => $bdweb_property['meta']['bedrooms'] ?? '',
                '_bathrooms' => $bdweb_property['meta']['bathsrooms'] ?? '',
            ]
        ];

        // Create the post
        error_log( "HPBDWEB: Creando post con datos: " . print_r( array_keys( $post_data ), true ) );
        
        // Debug: Log dirección específicamente
        $address = $bdweb_property['meta']['address'] ?? '';
        error_log( "HPBDWEB: Dirección a importar: '$address'" );
        error_log( "HPBDWEB: Metadatos direccion: direccion='{$post_data['meta_input']['direccion']}', _address='{$post_data['meta_input']['_address']}'" );
        
        error_log( "HPBDWEB: Llamando wp_insert_post..." );
        $post_id = wp_insert_post( $post_data );
        error_log( "HPBDWEB: wp_insert_post completado, resultado: " . (is_wp_error($post_id) ? 'ERROR' : $post_id) );

        if ( is_wp_error( $post_id ) ) {
            error_log( "HPBDWEB: Error creando post: " . $post_id->get_error_message() );
            return new WP_Error( 'post_creation_failed', __( 'Error al crear la propiedad: ', 'hp-bdweb-import' ) . $post_id->get_error_message() );
        }
        
        if ( ! $post_id ) {
            error_log( "HPBDWEB: wp_insert_post retornó 0" );
            return new WP_Error( 'post_creation_failed', __( 'Error al crear la propiedad: wp_insert_post retornó 0', 'hp-bdweb-import' ) );
        }
        
        error_log( "HPBDWEB: Post creado exitosamente con ID: $post_id" );

        // Handle taxonomies
        try {
            error_log( "HPBDWEB: Iniciando mapeo de taxonomías para post $post_id" );
            self::map_taxonomies( $post_id, $bdweb_property );
            error_log( "HPBDWEB: Taxonomías mapeadas exitosamente para post $post_id" );
    } catch ( \Exception $e ) {
            $error_msg = $e->getMessage();
            error_log( "HPBDWEB: Error en map_taxonomies: $error_msg" );
            return new WP_Error( 'taxonomy_mapping_failed', "Error en mapeo de taxonomías: $error_msg" );
        } catch ( Error $e ) {
            $error_msg = $e->getMessage();
            error_log( "HPBDWEB: Error fatal en map_taxonomies: $error_msg" );
            return new WP_Error( 'taxonomy_mapping_fatal', "Error fatal en mapeo de taxonomías: $error_msg" );
        }

        // Geocoding: obtener coordenadas si no las tiene
        try {
            error_log( "HPBDWEB: Iniciando geocoding para post $post_id" );
            self::handle_geocoding( $post_id, $bdweb_property );
            error_log( "HPBDWEB: Geocoding completado para post $post_id" );
    } catch ( \Exception $e ) {
            $error_msg = $e->getMessage();
            error_log( "HPBDWEB: Error en geocoding: $error_msg" );
            // Continuar aunque falle el geocoding - no es crítico
        }

        // Handle gallery images (optional)
        if ( $import_images ) {
            try {
                error_log( "HPBDWEB: Iniciando importación de imágenes para post $post_id" );
                self::import_gallery( $post_id, $bdweb_property );
                error_log( "HPBDWEB: Importación de imágenes completada para post $post_id" );
            } catch ( \Exception $e ) {
                $error_msg = $e->getMessage();
                error_log( "HPBDWEB: Error en importación de imágenes: $error_msg" );
                // Continuar aunque falle la importación de imágenes - no es crítico
            } catch ( Error $e ) {
                $error_msg = $e->getMessage();
                error_log( "HPBDWEB: Error fatal en importación de imágenes: $error_msg" );
                // Continuar aunque falle la importación de imágenes - no es crítico
            }
        }
        
        error_log( "HPBDWEB: === Todas las operaciones de importación completadas ===" );

        // Obtener log de importación de imágenes si existe
        $import_log = get_post_meta( $post_id, '_import_log', true );
        
            error_log( "HPBDWEB: === Preparando resultado de importación ===" );
            error_log( "HPBDWEB: Post ID creado: $post_id" );
            error_log( "HPBDWEB: Importación completa para post ID: $post_id" );
            
            return [
                'message' => sprintf( __( 'Propiedad importada exitosamente (ID: %d)', 'hp-bdweb-import' ), $post_id ),
                'post_id' => $post_id,
                'post_url' => get_permalink( $post_id ),
                'edit_url' => get_edit_post_link( $post_id ),
                'assigned_user' => get_userdata( $local_user_id )->display_name ?? __( 'Usuario desconocido', 'hp-bdweb-import' ),
                'import_log' => $import_log ? $import_log : 'No hay log de imágenes disponible'
            ];
            
    } catch ( \Exception $e ) {
            $error_msg = $e->getMessage();
            error_log( "HPBDWEB: Excepción en process_property_import: $error_msg" );
            return new WP_Error( 'import_exception', __( 'Excepción durante importación: ', 'hp-bdweb-import' ) . $error_msg );
        } catch ( Error $e ) {
            $error_msg = $e->getMessage();
            error_log( "HPBDWEB: Error fatal en process_property_import: $error_msg" );
            return new WP_Error( 'import_fatal_error', __( 'Error fatal durante importación: ', 'hp-bdweb-import' ) . $error_msg );
        }
    }

    /**
     * Find a local user by email address
     */
    private static function find_user_by_email( $email ) {
        if ( empty( $email ) ) {
            error_log( "HPBDWEB: Email vacío en find_user_by_email" );
            return false;
        }
        
        // Limpiar el email
        $email = trim( strtolower( $email ) );
        
        if ( ! is_email( $email ) ) {
            error_log( "HPBDWEB: Email inválido: $email" );
            return false;
        }

        error_log( "HPBDWEB: Buscando usuario con email: $email" );
        
        $user = get_user_by( 'email', $email );
        
        if ( $user ) {
            error_log( "HPBDWEB: Usuario encontrado: ID {$user->ID}, Display Name: {$user->display_name}" );
            return $user->ID;
        } else {
            error_log( "HPBDWEB: No se encontró usuario con email: $email" );
            return false;
        }
    }

    /**
     * Check if a property is already imported by title and author email
     */
    private static function is_property_already_imported( $title, $author_email ) {
        if ( empty( $title ) || empty( $author_email ) ) {
            return false;
        }

        // Find posts with the same title
        $existing_posts = get_posts( [
            'post_type' => 'properties',
            'title' => $title,
            'posts_per_page' => -1, // Get all matches
            'post_status' => 'any'
        ] );

        if ( empty( $existing_posts ) ) {
            return false;
        }

        // Check if any of these posts has the same author email
        foreach ( $existing_posts as $post ) {
            $post_author = get_userdata( $post->post_author );
            if ( $post_author && $post_author->user_email === $author_email ) {
                return true; // Same title + same author email = duplicate
            }
        }

        return false; // Same title but different authors = not duplicate
    }

    private static function map_taxonomies( $post_id, $bdweb_property ) {
        // Map purpose to transaccion taxonomy
        if ( ! empty( $bdweb_property['purpose'] ) && is_array( $bdweb_property['purpose'] ) ) {
            $purpose_names = wp_list_pluck( $bdweb_property['purpose'], 'nombre' );
            foreach ( $purpose_names as $purpose_name ) {
                self::assign_or_create_term( $post_id, $purpose_name, 'transaccion' );
            }
        }

        // Map property-type taxonomy
        if ( ! empty( $bdweb_property['property-type'] ) && is_array( $bdweb_property['property-type'] ) ) {
            $type_names = wp_list_pluck( $bdweb_property['property-type'], 'nombre' );
            foreach ( $type_names as $type_name ) {
                self::assign_or_create_term( $post_id, $type_name, 'property-type' );
            }
        }

        // Map amenidades taxonomy
        if ( ! empty( $bdweb_property['amenidades'] ) && is_array( $bdweb_property['amenidades'] ) ) {
            $amenity_names = wp_list_pluck( $bdweb_property['amenidades'], 'nombre' );
            foreach ( $amenity_names as $amenity_name ) {
                self::assign_or_create_term( $post_id, $amenity_name, 'amenidades' );
            }
        }

        // Map infraestructura taxonomy
        if ( ! empty( $bdweb_property['infraestructura'] ) && is_array( $bdweb_property['infraestructura'] ) ) {
            $infra_names = wp_list_pluck( $bdweb_property['infraestructura'], 'nombre' );
            foreach ( $infra_names as $infra_name ) {
                self::assign_or_create_term( $post_id, $infra_name, 'infraestructura' );
            }
        }

        // Handle location data as taxonomies (crear términos si no existen)
        if ( ! empty( $bdweb_property['estados'] ) && is_array( $bdweb_property['estados'] ) ) {
            $estado_name = $bdweb_property['estados'][0]['nombre'] ?? '';
            if ( $estado_name ) {
                self::assign_or_create_term( $post_id, $estado_name, 'estados' );
                update_post_meta( $post_id, 'estados', $estado_name );
            }
        }

        if ( ! empty( $bdweb_property['ciudad'] ) && is_array( $bdweb_property['ciudad'] ) ) {
            $ciudad_name = $bdweb_property['ciudad'][0]['nombre'] ?? '';
            if ( $ciudad_name ) {
                self::assign_or_create_term( $post_id, $ciudad_name, 'ciudad' );
                update_post_meta( $post_id, 'ciudad', $ciudad_name );
            }
        }

        if ( ! empty( $bdweb_property['colonia'] ) && is_array( $bdweb_property['colonia'] ) ) {
            $colonia_name = $bdweb_property['colonia'][0]['nombre'] ?? '';
            if ( $colonia_name ) {
                self::assign_or_create_term( $post_id, $colonia_name, 'colonia' );
                update_post_meta( $post_id, 'colonia', $colonia_name );
            }
        }

        // Handle amenities
        if ( ! empty( $bdweb_property['amenidades'] ) && is_array( $bdweb_property['amenidades'] ) ) {
            $amenity_names = wp_list_pluck( $bdweb_property['amenidades'], 'nombre' );
            update_post_meta( $post_id, '_amenities', $amenity_names );
        }
    }

    /**
     * Asigna un término a un post, creándolo si no existe
     */
    private static function assign_or_create_term( $post_id, $term_name, $taxonomy ) {
        if ( empty( $term_name ) || empty( $taxonomy ) ) {
            return false;
        }

        error_log( "HPBDWEB: Procesando término '$term_name' para taxonomía '$taxonomy'" );

        // Verificar si existe la taxonomía
        if ( ! taxonomy_exists( $taxonomy ) ) {
            error_log( "HPBDWEB: Taxonomía '$taxonomy' no existe, omitiendo..." );
            return false;
        }

        // Buscar término existente
        $existing_term = get_term_by( 'name', $term_name, $taxonomy );

        if ( $existing_term ) {
            // Término existe, asignarlo
            $result = wp_set_post_terms( $post_id, [ $existing_term->term_id ], $taxonomy, true );
            error_log( "HPBDWEB: Término existente '$term_name' asignado a post $post_id" );
            return $result;
        } else {
            // Término no existe, crearlo
            $new_term = wp_insert_term( $term_name, $taxonomy );
            
            if ( is_wp_error( $new_term ) ) {
                error_log( "HPBDWEB: Error creando término '$term_name' en '$taxonomy': " . $new_term->get_error_message() );
                return false;
            }

            // Asignar el nuevo término
            $result = wp_set_post_terms( $post_id, [ $new_term['term_id'] ], $taxonomy, true );
            error_log( "HPBDWEB: Nuevo término '$term_name' creado y asignado a post $post_id" );
            return $result;
        }
    }

    private static function import_gallery( $post_id, $bdweb_property ) {
        $import_log = [];
        
        // Import featured image
        if ( ! empty( $bdweb_property['featured_media'] ) ) {
            $import_log[] = "Intentando importar imagen destacada (ID: {$bdweb_property['featured_media']})";
            
            $media_response = wp_remote_get( 
                'https://bdwebmx.com/wp-json/wp/v2/media/' . $bdweb_property['featured_media'],
                [ 'timeout' => 15 ]
            );
            
            if ( is_wp_error( $media_response ) ) {
                $import_log[] = "Error en API de imagen destacada: " . $media_response->get_error_message();
            } else {
                $media_data = json_decode( wp_remote_retrieve_body( $media_response ), true );
                if ( ! empty( $media_data['source_url'] ) ) {
                    $import_log[] = "URL imagen destacada: " . $media_data['source_url'];
                    $attachment_id = self::import_image_from_url( $media_data['source_url'], $post_id );
                    if ( $attachment_id ) {
                        set_post_thumbnail( $post_id, $attachment_id );
                        $import_log[] = "✅ Imagen destacada importada (Attachment ID: $attachment_id)";
                    } else {
                        $import_log[] = "❌ Error importando imagen destacada";
                    }
                } else {
                    $import_log[] = "❌ No se encontró source_url en datos de imagen destacada";
                }
            }
        } else {
            $import_log[] = "No hay imagen destacada para importar";
        }

        // Debug: verificar estructura de datos
        $import_log[] = "Estructura de bdweb_property: " . print_r( array_keys( $bdweb_property ), true );
        if ( isset( $bdweb_property['meta'] ) ) {
            $import_log[] = "Estructura de meta: " . print_r( array_keys( $bdweb_property['meta'] ), true );
        }
        if ( isset( $bdweb_property['acf'] ) ) {
            $import_log[] = "Estructura de acf: " . print_r( array_keys( $bdweb_property['acf'] ), true );
        }

        // Import gallery images (optimizado para evitar timeouts)  
        // Según el JSON, la galería está en meta->gallery como array de IDs
        $gallery = null;
        if ( ! empty( $bdweb_property['meta']['gallery'] ) && is_array( $bdweb_property['meta']['gallery'] ) ) {
            $gallery = $bdweb_property['meta']['gallery'];
            $import_log[] = "Galería encontrada en meta->gallery con " . count($gallery) . " imágenes: " . implode(', ', $gallery);
        } else {
            $import_log[] = "No se encontró galería en meta->gallery";
        }
        
        if ( $gallery ) {
            $gallery_count = count( $gallery );
            $import_log[] = "Encontradas $gallery_count imágenes en galería, importando máximo 3";
            
            $gallery_ids = [];
            // Limitar a 3 imágenes para evitar timeouts
            foreach ( array_slice( $gallery, 0, 3 ) as $index => $image_id ) {
                $import_log[] = "Procesando imagen galería " . ($index + 1) . "/3 (ID: $image_id)";
                
                // Heartbeat cada imagen para mantener conexión viva
                self::send_heartbeat();
                
                $media_response = wp_remote_get( 
                    'https://bdwebmx.com/wp-json/wp/v2/media/' . $image_id,
                    [
                        'timeout' => 15, // Timeout más corto por imagen
                        'headers' => [
                            'User-Agent' => 'WordPress/' . get_bloginfo( 'version' ) . '; ' . home_url()
                        ]
                    ]
                );
                
                if ( is_wp_error( $media_response ) ) {
                    $import_log[] = "❌ Error en API imagen $image_id: " . $media_response->get_error_message();
                } else {
                    $media_data = json_decode( wp_remote_retrieve_body( $media_response ), true );
                    if ( ! empty( $media_data['source_url'] ) ) {
                        $import_log[] = "URL imagen $image_id: " . $media_data['source_url'];
                        $attachment_id = self::import_image_from_url( $media_data['source_url'], $post_id );
                        if ( $attachment_id ) {
                            $gallery_ids[] = $attachment_id;
                            $import_log[] = "✅ Imagen $image_id importada (Attachment ID: $attachment_id)";
                        } else {
                            $import_log[] = "❌ Error importando imagen $image_id";
                        }
                    } else {
                        $import_log[] = "❌ No source_url para imagen $image_id";
                    }
                }
                
                // Pequeña pausa entre imágenes para no sobrecargar
                if ( $index < 2 ) { // No pausar en la última imagen
                    usleep( 200000 ); // 0.2 segundos
                }
            }
            
            if ( ! empty( $gallery_ids ) ) {
                update_post_meta( $post_id, 'galeria_propiedad', $gallery_ids );
                $import_log[] = "✅ Galería guardada con " . count( $gallery_ids ) . " imágenes";
            } else {
                $import_log[] = "❌ No se pudo importar ninguna imagen de la galería";
            }
        } else {
            $import_log[] = "No hay galería de imágenes para importar";
        }
        
        // Guardar log de importación para debug
        update_post_meta( $post_id, '_import_log', implode( "\n", $import_log ) );
    }

    private static function import_image_from_url( $image_url, $post_id ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        // Verificar si la imagen ya existe para evitar duplicados
        $image_name = sanitize_file_name( basename( parse_url( $image_url, PHP_URL_PATH ) ) );
        $existing = get_posts([
            'post_type' => 'attachment',
            'meta_query' => [
                [
                    'key' => '_wp_attached_file',
                    'value' => $image_name,
                    'compare' => 'LIKE'
                ]
            ],
            'numberposts' => 1
        ]);
        
        if ( ! empty( $existing ) ) {
            return $existing[0]->ID;
        }

        // Configurar timeouts más agresivos para descarga de imágenes
        add_filter( 'http_request_timeout', function() { return 10; } );
        add_filter( 'http_request_args', function( $args ) {
            $args['timeout'] = 10;
            $args['redirection'] = 3;
            return $args;
        });

        $temp_file = download_url( $image_url );
        
        if ( is_wp_error( $temp_file ) ) {
            error_log( "Error descargando imagen $image_url: " . $temp_file->get_error_message() );
            return false;
        }

        // Verificar que el archivo se descargó correctamente
        if ( ! file_exists( $temp_file ) ) {
            error_log( "Archivo temporal no existe: $temp_file" );
            return false;
        }

        // Verificar tamaño del archivo para evitar imágenes muy grandes
        $file_size = filesize( $temp_file );
        if ( $file_size > 5 * MB_IN_BYTES ) { // Máximo 5MB
            error_log( "Imagen muy grande ($file_size bytes): $image_url" );
            @unlink( $temp_file );
            return false;
        }
        
        if ( $file_size < 1024 ) { // Mínimo 1KB
            error_log( "Imagen muy pequeña ($file_size bytes): $image_url" );
            @unlink( $temp_file );
            return false;
        }

        $file_array = [
            'name' => $image_name,
            'tmp_name' => $temp_file
        ];

        $attachment_id = media_handle_sideload( $file_array, $post_id );

        // Limpiar archivo temporal siempre
        if ( file_exists( $temp_file ) ) {
            @unlink( $temp_file );
        }

        if ( is_wp_error( $attachment_id ) ) {
            return false;
        }

        return $attachment_id;
    }
}
