<?php
namespace HPBDWEB;

if ( ! defined( 'ABSPATH' ) ) { exit; }

use HPBDWEB\Admin\Admin_Page;
use HPBDWEB\Admin\Ajax;

final class Plugin {

    private static $instance = null;

    public static function instance() : self {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action( 'admin_menu', [ $this, 'register_menu' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin' ] );

        // Allow ES Modules for our handle
        add_filter( 'script_loader_tag', [ $this, 'enable_module_type' ], 10, 3 );

        // AJAX handlers
        Ajax::register();
    }

    public function register_menu() {
        add_menu_page(
            __( 'Importar BDweb', 'hp-bdweb-import' ),
            __( 'Hunter Price Importación', 'hp-bdweb-import' ),
            HPBDWEB_CAP,
            HPBDWEB_SLUG,
            [ Admin_Page::class, 'render' ],
            'dashicons-database-import',
            59
        );
    }

    public function enqueue_admin( $hook ) {
        if ( $hook !== 'toplevel_page_' . HPBDWEB_SLUG ) {
            return;
        }

        // Timestamp dinámico para evitar caché completamente
        $dynamic_version = HPBDWEB_VER . '.' . time() . '.' . wp_rand(1000, 9999);
        
        // Debug info para desarrollo
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( "HPBDWEB: Cargando assets con versión anti-caché: {$dynamic_version}" );
        }

        // Styles (BEM) - Con timestamp para evitar caché
        wp_enqueue_style(
            'hpbdweb-admin',
            HPBDWEB_URL . 'assets/css/admin.css',
            [],
            $dynamic_version
        );

        // Scripts (module) - Con timestamp para evitar caché
        wp_enqueue_script(
            'hpbdweb-admin',
            HPBDWEB_URL . 'assets/js/admin.js',
            [],
            $dynamic_version,
            true
        );

        // Headers anti-caché para desarrollo (como hacen los plugins pro)
        add_action( 'wp_print_scripts', function() use ( $dynamic_version ) {
            ?>
            <script type="text/javascript">
                // Anti-cache para módulos ES6
                console.log('🔄 HPBDWEB Anti-Cache: <?php echo $dynamic_version; ?>');
                // Forzar recarga de módulos
                if ('serviceWorker' in navigator) {
                    navigator.serviceWorker.getRegistrations().then(function(registrations) {
                        registrations.forEach(function(registration) {
                            registration.unregister();
                        });
                    });
                }
            </script>
            <?php
        }, 5 );

        // Localize / settings for JS
        wp_localize_script( 'hpbdweb-admin', 'HPBDWEB_SETTINGS', [
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( HPBDWEB_NONCE ),
            // Buster específico para módulos ES6 anidados (ajax.js) para evitar caché del import()
            'module_buster' => $dynamic_version,
            'i18n'     => [
                'testing'            => __( 'Probando…', 'hp-bdweb-import' ),
                'loading'            => __( 'Cargando…', 'hp-bdweb-import' ),
                'ok'                 => __( '¡Solicitud AJAX OK!', 'hp-bdweb-import' ),
                'error'              => __( 'Error en la solicitud AJAX', 'hp-bdweb-import' ),
                'api_connected'      => __( 'API conectada correctamente', 'hp-bdweb-import' ),
                'api_error'          => __( 'Error conectando con la API', 'hp-bdweb-import' ),
                'importing'          => __( 'Importando…', 'hp-bdweb-import' ),
                'import_success'     => __( 'Propiedad importada exitosamente', 'hp-bdweb-import' ),
                'import_error'       => __( 'Error importando propiedad', 'hp-bdweb-import' ),
                'already_imported'   => __( 'Esta propiedad ya fue importada', 'hp-bdweb-import' ),
                'no_properties'      => __( 'No se encontraron propiedades', 'hp-bdweb-import' ),
            ],
        ] );
    }

    public function enable_module_type( $tag, $handle, $src ) {
        if ( $handle === 'hpbdweb-admin' ) {
            // Headers anti-caché para módulos ES6 (método profesional)
            $cache_buster = time() . wp_rand(1000, 9999);
            $src_with_cache_buster = add_query_arg( 'nocache', $cache_buster, $src );
            
            $tag = '<script type="module" src="' . esc_url( $src_with_cache_buster ) . '" crossorigin></script>';
            
            // Debug para desarrollo
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( "HPBDWEB: Módulo ES6 cargado con anti-caché: {$src_with_cache_buster}" );
            }
        }
        return $tag;
    }
}
