<?php
// Asegurar includes para funciones de medios
require_once ABSPATH . 'wp-admin/includes/image.php';
require_once ABSPATH . 'wp-admin/includes/file.php';
require_once ABSPATH . 'wp-admin/includes/media.php';

/**
 * Función de importación de una propiedad desde el objeto Tokko.
 */
function tbi_importar_una_propiedad_desde_tokko( $tokko_obj ) {
    // Log para debug
    error_log('TBI DEBUG: Iniciando importación de propiedad. ID: ' . ($tokko_obj['id'] ?? 'NO_ID'));
    
    $tokko_id = intval( $tokko_obj['id'] );
    if (!$tokko_id) {
        error_log('TBI ERROR: ID de Tokko inválido');
        return false;
    }
    
    $existing = get_posts([
        'post_type'  => 'properties',
        'meta_key'   => '_tokko_id',
        'meta_value' => $tokko_id,
        'fields'     => 'ids',
        'numberposts'=> 1
    ]);
    if ( ! empty( $existing ) ) {
        error_log('TBI DEBUG: Propiedad ya existe con ID: ' . $tokko_id);
        return false;
    }

    // Crear el post
    $post_data = [
        'post_title'   => sanitize_text_field( $tokko_obj['publication_title'] ?? 'Propiedad sin título' ),
        'post_content' => sanitize_textarea_field( $tokko_obj['description'] ?? $tokko_obj['rich_description'] ?? '' ),
        'post_type'    => 'properties',
        'post_status'  => 'publish',
        'post_author'  => get_current_user_id(),
    ];
    
    $post_id = wp_insert_post( $post_data );
    if ( is_wp_error( $post_id ) ) {
        error_log('TBI ERROR: No se pudo crear el post: ' . $post_id->get_error_message());
        return false;
    }
    
    error_log('TBI DEBUG: Post creado con ID: ' . $post_id);

    // Guardar ID de Tokko
    update_post_meta( $post_id, '_tokko_id', $tokko_id );

    // Precio y moneda
    if ( ! empty( $tokko_obj['operations'][0]['prices'][0]['price'] ) ) {
        $price = floatval( $tokko_obj['operations'][0]['prices'][0]['price'] );
        update_post_meta( $post_id, 'price', $price );
        error_log('TBI DEBUG: Precio guardado: ' . $price);
    }

    if ( ! empty( $tokko_obj['operations'][0]['prices'][0]['currency'] ) ) {
        $currency = sanitize_text_field( $tokko_obj['operations'][0]['prices'][0]['currency'] );
        update_post_meta( $post_id, 'currency', $currency );
        error_log('TBI DEBUG: Moneda guardada: ' . $currency);
    }

    // Taxonomías principales
    if ( ! empty( $tokko_obj['operations'][0]['operation_type'] ) ) {
        $operation = sanitize_text_field( $tokko_obj['operations'][0]['operation_type'] );
        wp_set_object_terms( $post_id, $operation, 'purpose', false );
        error_log('TBI DEBUG: Operación guardada: ' . $operation);
    }

    if ( ! empty( $tokko_obj['type']['name'] ) ) {
        $property_type = sanitize_text_field( $tokko_obj['type']['name'] );
        wp_set_object_terms( $post_id, $property_type, 'property-type', false );
        error_log('TBI DEBUG: Tipo de propiedad guardado: ' . $property_type);
    }

    // Mapeo completo de campos numéricos
    $numeric_fields = [
        'bathroom_amount'     => 'bathrooms',      // Era 'bathsrooms' - corregido
        'toilet_amount'       => 'half-baths',
        'room_amount'         => 'bedrooms',
        'parking_lot_amount'  => 'garaje',
        'floors_amount'       => 'total-floors',
        'front_measure'       => 'frente',
        'roofed_surface'      => 'techado',
        'surface'             => 'superficie-total',
        'total_surface'       => 'superficie-construida',
        'age'                 => 'antiguedad',
        'expenses'            => 'expenses',
        'suite_amount'        => 'suites',
        'living_amount'       => 'salas',
        'dining_room'         => 'comedores',
        'tv_rooms'            => 'cuartos-tv',
        'covered_parking_lot' => 'cocheras-techadas',
        'uncovered_parking_lot' => 'cocheras-descubiertas',
    ];
    
    foreach ($numeric_fields as $tokko_key => $meta_key) {
        if ( isset( $tokko_obj[$tokko_key] ) && $tokko_obj[$tokko_key] !== '' ) {
            $value = is_numeric($tokko_obj[$tokko_key]) ? floatval($tokko_obj[$tokko_key]) : sanitize_text_field($tokko_obj[$tokko_key]);
            update_post_meta( $post_id, $meta_key, $value );
            error_log('TBI DEBUG: Meta guardado: ' . $meta_key . ' = ' . $value);
        }
    }

    // Campos de texto adicionales
    $text_fields = [
        'address'                => 'direccion-real',
        'fake_address'           => 'direccion-publica',
        'reference_code'         => 'codigo-referencia',
        'property_condition'     => 'condicion',
        'situation'              => 'situacion',
        'orientation'            => 'orientacion',
        'public_url'             => 'url-publica',
        'real_address'           => 'direccion-completa',
    ];
    
    foreach ($text_fields as $tokko_key => $meta_key) {
        if ( !empty( $tokko_obj[$tokko_key] ) ) {
            update_post_meta( $post_id, $meta_key, sanitize_text_field( $tokko_obj[$tokko_key] ) );
            error_log('TBI DEBUG: Texto guardado: ' . $meta_key . ' = ' . $tokko_obj[$tokko_key]);
        }
    }

    // Manejo mejorado de ubicación
    if ( ! empty( $tokko_obj['location'] ) ) {
        $location = $tokko_obj['location'];
        
        // Guardar ubicación completa
        if ( !empty( $location['full_location'] ) ) {
            update_post_meta( $post_id, 'ubicacion-completa', sanitize_text_field( $location['full_location'] ) );
            
            // Parsear ubicación: "México | Edo. de México | Zinacantepec | San Miguel Zinacantepec"
            $parts = explode( ' | ', $location['full_location'] );
            error_log('TBI DEBUG: Partes de ubicación: ' . print_r($parts, true));
            
            if ( isset( $parts[0] ) ) { // País
                $term = wp_insert_term( sanitize_text_field( $parts[0] ), 'pais' );
                if ( ! is_wp_error( $term ) )
                    wp_set_object_terms( $post_id, intval($term['term_id']), 'pais', false );
            }
            if ( isset( $parts[1] ) ) { // Estado
                $term = wp_insert_term( sanitize_text_field( $parts[1] ), 'estados' );
                if ( ! is_wp_error( $term ) )
                    wp_set_object_terms( $post_id, intval($term['term_id']), 'estados', false );
                error_log('TBI DEBUG: Estado guardado: ' . $parts[1]);
            }
            if ( isset( $parts[2] ) ) { // Ciudad/Municipio
                $term = wp_insert_term( sanitize_text_field( $parts[2] ), 'ciudad' );
                if ( ! is_wp_error( $term ) )
                    wp_set_object_terms( $post_id, intval($term['term_id']), 'ciudad', false );
                error_log('TBI DEBUG: Ciudad guardada: ' . $parts[2]);
            }
            if ( isset( $parts[3] ) ) { // Colonia/Barrio
                $term = wp_insert_term( sanitize_text_field( $parts[3] ), 'colonia' );
                if ( ! is_wp_error( $term ) )
                    wp_set_object_terms( $post_id, intval($term['term_id']), 'colonia', false );
                error_log('TBI DEBUG: Colonia guardada: ' . $parts[3]);
            }
        }
        
        // Ubicación corta
        if ( !empty( $location['short_location'] ) ) {
            update_post_meta( $post_id, 'ubicacion-corta', sanitize_text_field( $location['short_location'] ) );
        }
        
        // Código postal si existe
        if ( !empty( $location['zip_code'] ) ) {
            update_post_meta( $post_id, 'codigo-postal', sanitize_text_field( $location['zip_code'] ) );
        }
    }

    // Coordenadas geográficas
    if ( isset( $tokko_obj['geo_lat'] ) && isset( $tokko_obj['geo_long'] ) ) {
        update_post_meta( $post_id, '56d95c68038fa32527b1d45f2b7de06e_lat', sanitize_text_field( $tokko_obj['geo_lat'] ) );
        update_post_meta( $post_id, '56d95c68038fa32527b1d45f2b7de06e_lng', sanitize_text_field( $tokko_obj['geo_long'] ) );
        
        // También como campos simples
        update_post_meta( $post_id, 'latitud', sanitize_text_field( $tokko_obj['geo_lat'] ) );
        update_post_meta( $post_id, 'longitud', sanitize_text_field( $tokko_obj['geo_long'] ) );
        
        error_log('TBI DEBUG: Coordenadas guardadas: ' . $tokko_obj['geo_lat'] . ', ' . $tokko_obj['geo_long']);
    }

    // Información del agente/productor
    if ( !empty( $tokko_obj['producer'] ) ) {
        $producer = $tokko_obj['producer'];
        if ( !empty( $producer['name'] ) )
            update_post_meta( $post_id, 'agente-nombre', sanitize_text_field( $producer['name'] ) );
        if ( !empty( $producer['email'] ) )
            update_post_meta( $post_id, 'agente-email', sanitize_email( $producer['email'] ) );
        if ( !empty( $producer['phone'] ) )
            update_post_meta( $post_id, 'agente-telefono', sanitize_text_field( $producer['phone'] ) );
        if ( !empty( $producer['cellphone'] ) )
            update_post_meta( $post_id, 'agente-celular', sanitize_text_field( $producer['cellphone'] ) );
        if ( !empty( $producer['picture'] ) )
            update_post_meta( $post_id, 'agente-foto', esc_url_raw( $producer['picture'] ) );
        
        error_log('TBI DEBUG: Agente guardado: ' . ($producer['name'] ?? 'Sin nombre'));
    }

    // Información de la sucursal/branch
    if ( !empty( $tokko_obj['branch'] ) ) {
        $branch = $tokko_obj['branch'];
        if ( !empty( $branch['name'] ) )
            update_post_meta( $post_id, 'sucursal-nombre', sanitize_text_field( $branch['name'] ) );
        if ( !empty( $branch['address'] ) )
            update_post_meta( $post_id, 'sucursal-direccion', sanitize_text_field( $branch['address'] ) );
        if ( !empty( $branch['phone'] ) )
            update_post_meta( $post_id, 'sucursal-telefono', sanitize_text_field( $branch['phone'] ) );
        if ( !empty( $branch['email'] ) )
            update_post_meta( $post_id, 'sucursal-email', sanitize_email( $branch['email'] ) );
        if ( !empty( $branch['logo'] ) )
            update_post_meta( $post_id, 'sucursal-logo', esc_url_raw( $branch['logo'] ) );
    }

    if ( ! empty( $tokko_obj['tags'] ) && is_array( $tokko_obj['tags'] ) ) {
        foreach ( $tokko_obj['tags'] as $tag ) {
            $name = sanitize_text_field( $tag['name'] );
            switch ( intval( $tag['type'] ) ) {
                case 1: if ( taxonomy_exists( 'services' ) ) wp_set_object_terms( $post_id, $name, 'services', true ); break;
                case 2: wp_set_object_terms( $post_id, $name, 'amenidades', true ); break;
                case 3: wp_set_object_terms( $post_id, $name, 'infraestructura', true ); break;
            }
        }
    }

    if ( ! empty( $tokko_obj['photos'] ) && is_array( $tokko_obj['photos'] ) ) {
        $gallery = [];
        foreach ( $tokko_obj['photos'] as $i => $p ) {
            if ( empty( $p['original'] ) ) continue;
            $img_id = media_sideload_image( esc_url_raw( $p['original'] ), $post_id, null, 'id' );
            if ( ! is_wp_error($img_id) ) {
                if ( $i === 0 ) set_post_thumbnail( $post_id, $img_id );
                else $gallery[] = $img_id;
            }
        }
        if ( $gallery ) update_post_meta( $post_id, 'gallery', implode(',', $gallery) );
    }

    // Fechas y datos adicionales
    if ( ! empty( $tokko_obj['created_at'] ) ) {
        update_post_meta( $post_id, 'tokko_created_at', sanitize_text_field( $tokko_obj['created_at'] ) );
        // Convertir fecha para WordPress
        $date_wp = date('Y-m-d H:i:s', strtotime($tokko_obj['created_at']));
        update_post_meta( $post_id, 'fecha-creacion', $date_wp );
    }

    // Datos adicionales importantes
    $additional_fields = [
        'status'                     => 'estado-tokko',
        'quality_level'              => 'nivel-calidad',
        'web_price'                  => 'mostrar-precio-web',
        'is_starred_on_web'          => 'destacada',
        'has_temporary_rent'         => 'renta-temporal',
        'legally_checked'            => 'revision-legal',
        'seo_description'            => 'descripcion-seo',
        'seo_keywords'               => 'palabras-clave-seo',
        'transaction_requirements'   => 'requisitos-transaccion',
    ];
    
    foreach ($additional_fields as $tokko_key => $meta_key) {
        if ( isset( $tokko_obj[$tokko_key] ) && $tokko_obj[$tokko_key] !== '' ) {
            $value = is_bool($tokko_obj[$tokko_key]) ? ($tokko_obj[$tokko_key] ? 'yes' : 'no') : sanitize_text_field($tokko_obj[$tokko_key]);
            update_post_meta( $post_id, $meta_key, $value );
        }
    }

    error_log('TBI DEBUG: Importación completada para post ID: ' . $post_id);
    return true;
}
